jQuery(document).ready(function($) {
    'use strict';

    var modifiedProducts = {};
    var currentPage = 1;
    var totalPages = 1;

    function loadProducts() {
        var category = $('#woonutri-filter-category').val();
        var status = $('#woonutri-filter-status').val();
        var search = $('#woonutri-filter-search').val();

        $('#woonutri-bulk-tbody').html(
            '<tr><td colspan="20" class="bulk-loading">' +
            '<span class="spinner is-active"></span> Loading products...</td></tr>'
        );

        $.ajax({
            url: woonutriBulk.ajax_url,
            type: 'POST',
            data: {
                action: 'woonutri_get_products_data',
                nonce: woonutriBulk.nonce,
                category: category,
                status: status,
                search: search,
                page: currentPage
            },
            success: function(response) {
                if (response.success) {
                    renderProducts(response.data.products);
                    totalPages = response.data.pages;
                    updatePagination(response.data);
                } else {
                    showError(response.data.message);
                }
            },
            error: function() {
                showError(woonutriBulk.i18n.error);
            }
        });
    }

    function renderProducts(products) {
        var tbody = $('#woonutri-bulk-tbody');
        tbody.empty();

        if (products.length === 0) {
            tbody.html(
                '<tr class="no-data"><td colspan="20">' +
                'No products found matching your criteria</td></tr>'
            );
            return;
        }

        var fields = woonutriBulk.fields;

        products.forEach(function(product) {
            var row = $('<tr data-product-id="' + product.id + '"></tr>');

            row.append('<td class="col-select"><input type="checkbox" class="product-checkbox" value="' + product.id + '"></td>');

            var thumb = product.thumbnail || '';
            var thumbHtml = thumb ? '<img src="' + thumb + '" class="product-thumb" alt="">' : '<div class="product-thumb"></div>';

            row.append(
                '<td class="col-product">' +
                '<div class="product-cell">' +
                thumbHtml +
                '<div class="product-info">' +
                '<div class="product-name" title="' + escapeHtml(product.title) + '">' + escapeHtml(product.title) + '</div>' +
                '<div class="product-link"><a href="' + product.edit_url + '" target="_blank">Edit</a></div>' +
                '</div></div></td>'
            );

            fields.forEach(function(field) {
                var value = product.nutrition[field] || '';
                var inputType = (field === 'serving_size' || field === 'servings_per_container') ? 'text' : 'text';
                row.append(
                    '<td class="col-field">' +
                    '<input type="' + inputType + '" ' +
                    'class="field-input" ' +
                    'data-field="' + field + '" ' +
                    'data-original="' + escapeHtml(value) + '" ' +
                    'value="' + escapeHtml(value) + '">' +
                    '</td>'
                );
            });

            row.append(
                '<td class="col-actions">' +
                '<button type="button" class="button button-small save-row" title="Save this row">' +
                '<span class="dashicons dashicons-saved"></span></button>' +
                '</td>'
            );

            tbody.append(row);
        });
    }

    function updatePagination(data) {
        var container = $('.woonutri-bulk-pagination');
        if (container.length === 0) {
            container = $('<div class="woonutri-bulk-pagination"></div>');
            $('.woonutri-bulk-table-container').append(container);
        }

        if (data.pages <= 1) {
            container.hide();
            return;
        }

        container.show().html(
            '<button type="button" class="button prev-page" ' + (currentPage <= 1 ? 'disabled' : '') + '>&laquo; Prev</button>' +
            '<span class="page-info">Page ' + currentPage + ' of ' + data.pages + ' (' + data.total + ' products)</span>' +
            '<button type="button" class="button next-page" ' + (currentPage >= data.pages ? 'disabled' : '') + '>Next &raquo;</button>'
        );
    }

    function escapeHtml(text) {
        if (!text) return '';
        var div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    function showError(message) {
        $('#woonutri-bulk-tbody').html(
            '<tr class="no-data"><td colspan="20" style="color:#ef4444;">' + message + '</td></tr>'
        );
    }

    function showStatus(message, type) {
        var status = $('#woonutri-save-status');
        status.removeClass('saving saved error').addClass(type).text(message);

        if (type === 'saved') {
            setTimeout(function() {
                status.text('');
            }, 3000);
        }
    }

    function collectModifiedData() {
        var products = [];

        $('#woonutri-bulk-tbody tr').each(function() {
            var row = $(this);
            var productId = row.data('product-id');
            if (!productId) return;

            var hasChanges = false;
            var nutrition = {};

            row.find('.field-input').each(function() {
                var input = $(this);
                var field = input.data('field');
                var value = input.val();
                var original = input.data('original');

                if (value !== original) {
                    hasChanges = true;
                }
                nutrition[field] = value;
            });

            if (hasChanges) {
                products.push({
                    id: productId,
                    nutrition: nutrition
                });
            }
        });

        return products;
    }

    function saveAllChanges() {
        var products = collectModifiedData();

        if (products.length === 0) {
            showStatus('No changes to save', 'saved');
            return;
        }

        showStatus(woonutriBulk.i18n.saving, 'saving');

        $.ajax({
            url: woonutriBulk.ajax_url,
            type: 'POST',
            data: {
                action: 'woonutri_bulk_save',
                nonce: woonutriBulk.nonce,
                products: products
            },
            success: function(response) {
                if (response.success) {
                    showStatus(response.data.message, 'saved');

                    $('#woonutri-bulk-tbody .field-input').each(function() {
                        var input = $(this);
                        input.data('original', input.val()).removeClass('modified');
                    });
                    $('#woonutri-bulk-tbody tr').removeClass('modified');
                } else {
                    showStatus(response.data.message, 'error');
                }
            },
            error: function() {
                showStatus(woonutriBulk.i18n.error, 'error');
            }
        });
    }

    function exportCSV() {
        var selectedIds = [];
        $('.product-checkbox:checked').each(function() {
            selectedIds.push($(this).val());
        });

        showStatus('Exporting...', 'saving');

        $.ajax({
            url: woonutriBulk.ajax_url,
            type: 'POST',
            data: {
                action: 'woonutri_bulk_export',
                nonce: woonutriBulk.nonce,
                product_ids: selectedIds
            },
            success: function(response) {
                if (response.success) {
                    downloadCSV(response.data.csv, response.data.filename);
                    showStatus(woonutriBulk.i18n.export_success, 'saved');
                } else {
                    showStatus(response.data.message, 'error');
                }
            },
            error: function() {
                showStatus(woonutriBulk.i18n.error, 'error');
            }
        });
    }

    function downloadCSV(data, filename) {
        var csv = data.map(function(row) {
            return row.map(function(cell) {
                if (typeof cell === 'string' && (cell.indexOf(',') !== -1 || cell.indexOf('"') !== -1)) {
                    return '"' + cell.replace(/"/g, '""') + '"';
                }
                return cell;
            }).join(',');
        }).join('\n');

        var blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
        var link = document.createElement('a');
        var url = URL.createObjectURL(blob);

        link.setAttribute('href', url);
        link.setAttribute('download', filename);
        link.style.visibility = 'hidden';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    }

    function importCSV(file) {
        var reader = new FileReader();

        reader.onload = function(e) {
            var csv = e.target.result;
            var lines = csv.split(/\r\n|\n/);
            var data = [];

            lines.forEach(function(line) {
                if (line.trim() === '') return;

                var row = [];
                var insideQuote = false;
                var field = '';

                for (var i = 0; i < line.length; i++) {
                    var char = line[i];

                    if (char === '"') {
                        if (insideQuote && line[i + 1] === '"') {
                            field += '"';
                            i++;
                        } else {
                            insideQuote = !insideQuote;
                        }
                    } else if (char === ',' && !insideQuote) {
                        row.push(field);
                        field = '';
                    } else {
                        field += char;
                    }
                }
                row.push(field);
                data.push(row);
            });

            if (data.length < 2) {
                showStatus('CSV file is empty or invalid', 'error');
                return;
            }

            showStatus('Importing...', 'saving');

            $.ajax({
                url: woonutriBulk.ajax_url,
                type: 'POST',
                data: {
                    action: 'woonutri_bulk_import',
                    nonce: woonutriBulk.nonce,
                    csv_data: data
                },
                success: function(response) {
                    if (response.success) {
                        showStatus(response.data.message, 'saved');
                        loadProducts();
                    } else {
                        showStatus(response.data.message, 'error');
                    }
                },
                error: function() {
                    showStatus(woonutriBulk.i18n.error, 'error');
                }
            });
        };

        reader.readAsText(file);
    }

    function applyTemplate() {
        var templateKey = $('#woonutri-template-select').val();
        if (!templateKey) {
            alert('Please select a template');
            return;
        }

        var selectedIds = [];
        $('.product-checkbox:checked').each(function() {
            selectedIds.push(parseInt($(this).val()));
        });

        if (selectedIds.length === 0) {
            alert(woonutriBulk.i18n.no_products);
            return;
        }

        if (!confirm(woonutriBulk.i18n.confirm_template)) {
            return;
        }

        showStatus('Applying template...', 'saving');

        $.ajax({
            url: woonutriBulk.ajax_url,
            type: 'POST',
            data: {
                action: 'woonutri_apply_template',
                nonce: woonutriBulk.nonce,
                template: templateKey,
                product_ids: selectedIds
            },
            success: function(response) {
                if (response.success) {
                    showStatus(response.data.message, 'saved');
                    loadProducts();
                } else {
                    showStatus(response.data.message, 'error');
                }
            },
            error: function() {
                showStatus(woonutriBulk.i18n.error, 'error');
            }
        });
    }

    $('#woonutri-load-products').on('click', function() {
        currentPage = 1;
        loadProducts();
    });

    $('#woonutri-save-all').on('click', saveAllChanges);

    $('#woonutri-export-csv').on('click', exportCSV);

    $('#woonutri-import-csv').on('click', function() {
        $('#woonutri-csv-file').trigger('click');
    });

    $('#woonutri-csv-file').on('change', function() {
        var file = this.files[0];
        if (file) {
            importCSV(file);
            this.value = '';
        }
    });

    $('#woonutri-apply-template').on('click', applyTemplate);

    $('#woonutri-select-all').on('click', function() {
        $('.product-checkbox').prop('checked', true);
        $('#woonutri-check-all').prop('checked', true);
    });

    $('#woonutri-deselect-all').on('click', function() {
        $('.product-checkbox').prop('checked', false);
        $('#woonutri-check-all').prop('checked', false);
    });

    $(document).on('change', '#woonutri-check-all', function() {
        $('.product-checkbox').prop('checked', $(this).is(':checked'));
    });

    $(document).on('input', '.field-input', function() {
        var input = $(this);
        var original = input.data('original');
        var current = input.val();

        if (current !== original) {
            input.addClass('modified');
            input.closest('tr').addClass('modified');
        } else {
            input.removeClass('modified');
            var row = input.closest('tr');
            if (row.find('.field-input.modified').length === 0) {
                row.removeClass('modified');
            }
        }
    });

    $(document).on('click', '.save-row', function() {
        var row = $(this).closest('tr');
        var productId = row.data('product-id');
        var nutrition = {};

        row.find('.field-input').each(function() {
            var input = $(this);
            nutrition[input.data('field')] = input.val();
        });

        var button = $(this);
        button.prop('disabled', true);

        $.ajax({
            url: woonutriBulk.ajax_url,
            type: 'POST',
            data: {
                action: 'woonutri_bulk_save',
                nonce: woonutriBulk.nonce,
                products: [{ id: productId, nutrition: nutrition }]
            },
            success: function(response) {
                if (response.success) {
                    row.find('.field-input').each(function() {
                        var input = $(this);
                        input.data('original', input.val()).removeClass('modified');
                    });
                    row.removeClass('modified');

                    button.find('.dashicons').removeClass('dashicons-saved').addClass('dashicons-yes');
                    setTimeout(function() {
                        button.find('.dashicons').removeClass('dashicons-yes').addClass('dashicons-saved');
                    }, 1500);
                }
                button.prop('disabled', false);
            },
            error: function() {
                button.prop('disabled', false);
            }
        });
    });

    $(document).on('click', '.prev-page', function() {
        if (currentPage > 1) {
            currentPage--;
            loadProducts();
        }
    });

    $(document).on('click', '.next-page', function() {
        if (currentPage < totalPages) {
            currentPage++;
            loadProducts();
        }
    });

    $(document).on('keydown', '.field-input', function(e) {
        if (e.key === 'Tab') {
            return;
        }

        if (e.key === 'Enter') {
            e.preventDefault();
            var inputs = $('#woonutri-bulk-tbody .field-input');
            var currentIndex = inputs.index(this);
            var cols = woonutriBulk.fields.length;

            if (e.shiftKey) {
                if (currentIndex >= cols) {
                    inputs.eq(currentIndex - cols).focus();
                }
            } else {
                if (currentIndex + cols < inputs.length) {
                    inputs.eq(currentIndex + cols).focus();
                }
            }
        }

        if (e.key === 'ArrowDown') {
            e.preventDefault();
            var inputs = $('#woonutri-bulk-tbody .field-input');
            var currentIndex = inputs.index(this);
            var cols = woonutriBulk.fields.length;
            if (currentIndex + cols < inputs.length) {
                inputs.eq(currentIndex + cols).focus();
            }
        }

        if (e.key === 'ArrowUp') {
            e.preventDefault();
            var inputs = $('#woonutri-bulk-tbody .field-input');
            var currentIndex = inputs.index(this);
            var cols = woonutriBulk.fields.length;
            if (currentIndex >= cols) {
                inputs.eq(currentIndex - cols).focus();
            }
        }
    });

    $(document).on('keydown', function(e) {
        if (e.ctrlKey && e.key === 's') {
            e.preventDefault();
            saveAllChanges();
        }
    });
});
