
(function ($) {
    'use strict';
    const config = {

        animationDuration: 600,
        staggerDelay: 50,
        expandDuration: 300,
        labelSelector: '.woonutri-label',
        tableSelector: '.woonutri-table',
        rowSelector: '.woonutri-table tr',
        expandableClass: 'woonutri-expandable',
        expandedClass: 'woonutri-expanded',
        detailsClass: 'woonutri-details',
        animatedClass: 'woonutri-animated',
        ariaExpanded: 'aria-expanded',
        ariaControls: 'aria-controls',
        ariaHidden: 'aria-hidden',
        ariaLabel: 'aria-label',
        role: 'role',
        expandAllText: 'Expand All',
        collapseAllText: 'Collapse All'
    };

    
    function init() {
        const $labels = $(config.labelSelector);

        if (!$labels.length) return;
        addStyles();
        $labels.each(function (index) {
            const $label = $(this);
            enhanceAccessibility($label);
            makeExpandable($label);
            addHoverEffects($label);
            setTimeout(() => {
                revealLabel($label);
            }, index * 100);
        });
    }

    
    function addStyles() {
        const styles = `
            <style id="woonutri-frontend-styles">
                
                .woonutri-label {
                    opacity: 0;
                    transform: translateY(20px);
                    transition: opacity ${config.animationDuration}ms ease-out,
                                transform ${config.animationDuration}ms ease-out;
                }
                
                .woonutri-label.woonutri-animated {
                    opacity: 1;
                    transform: translateY(0);
                }
                
                
                .woonutri-table tr {
                    opacity: 0;
                    transform: translateX(-10px);
                    transition: opacity 400ms ease-out,
                                transform 400ms ease-out,
                                background-color 300ms ease;
                }
                
                .woonutri-table tr.woonutri-animated {
                    opacity: 1;
                    transform: translateX(0);
                }
                
                
                .woonutri-expandable {
                    cursor: pointer;
                    position: relative;
                    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
                }
                
                .woonutri-expandable:hover {
                    transform: translateY(-1px);
                    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
                }
                
                .woonutri-details {
                    max-height: 0;
                    overflow: hidden;
                    transition: max-height 0.4s cubic-bezier(0.4, 0, 0.2, 1),
                                opacity 0.3s ease,
                                padding 0.3s ease;
                    opacity: 0;
                    padding: 0;
                }
                
                .woonutri-details.woonutri-expanded {
                    max-height: 800px;
                    opacity: 1;
                    padding: 0.5rem 0;
                }
                
                
                .woonutri-inner-table {
                    transform: translateY(-10px);
                    transition: transform 0.3s ease 0.1s;
                }
                
                .woonutri-details.woonutri-expanded .woonutri-inner-table {
                    transform: translateY(0);
                }
                
                
                .woonutri-table tr:hover {
                    background-color: rgba(0, 0, 0, 0.03);
                    transition: background-color 0.2s ease;
                }
                
                
                .woonutri-expandable:focus {
                    outline: 2px solid rgba(42, 122, 226, 0.5);
                    outline-offset: -2px;
                    box-shadow: 0 0 0 4px rgba(42, 122, 226, 0.1);
                }
                
                
                .woonutri-loading {
                    position: relative;
                    pointer-events: none;
                }
                
                .woonutri-loading:after {
                    content: '';
                    position: absolute;
                    top: 50%;
                    left: 50%;
                    width: 20px;
                    height: 20px;
                    margin: -10px 0 0 -10px;
                    border: 2px solid rgba(42, 122, 226, 0.2);
                    border-top: 2px solid rgba(42, 122, 226, 0.8);
                    border-radius: 50%;
                    animation: woonutri-spin 1s linear infinite;
                }
                
                @keyframes woonutri-spin {
                    0% { transform: rotate(0deg); }
                    100% { transform: rotate(360deg); }
                }
                
                
                @media print {
                    .woonutri-label,
                    .woonutri-table tr {
                        opacity: 1 !important;
                        transform: none !important;
                    }
                    
                    .woonutri-details {
                        max-height: none !important;
                        overflow: visible !important;
                        opacity: 1 !important;
                        padding: 0.5rem 0 !important;
                    }
                    
                    .woonutri-expandable td:after {
                        display: none !important;
                    }
                }
                
                
                @media (prefers-reduced-motion: reduce) {
                    .woonutri-label,
                    .woonutri-table tr,
                    .woonutri-expandable,
                    .woonutri-details,
                    .woonutri-inner-table {
                        transition: none !important;
                        animation: none !important;
                        transform: none !important;
                    }
                }
            </style>
        `;
        if (!$('#woonutri-frontend-styles').length) {
            $('head').append(styles);
        }
    }

    
    function enhanceAccessibility($label) {

        $label.attr(config.role, 'region')
            .attr(config.ariaLabel, 'Nutrition Facts');
        const $table = $label.find(config.tableSelector);
        if (!$table.attr('role')) {
            $table.attr(config.role, 'table');
            $table.find('tr').attr(config.role, 'row');
            $table.find('td:first-child').attr(config.role, 'rowheader');
            $table.find('td:not(:first-child)').attr(config.role, 'cell');
        }
    }

    
    function makeExpandable($label) {

        const $existingExpandable = $label.find('.woonutri-expandable');

        if ($existingExpandable.length > 0) {

            $existingExpandable.each(function () {
                const $header = $(this);
                const $detailsRow = $header.next('tr');
                const $details = $detailsRow.find('.woonutri-details');

                if ($details.length > 0) {

                    $detailsRow.addClass('woonutri-details-row');
                    $details.parent('td').addClass('woonutri-details-cell');
                    $header.attr('tabindex', '0')
                        .attr(config.ariaExpanded, 'false')
                        .attr(config.role, 'button');

                    $details.attr(config.ariaHidden, 'true');
                    $header.off('click keypress').on('click keypress', function (e) {

                        if (e.type === 'keypress' && e.which !== 13 && e.which !== 32) {
                            return;
                        }

                        e.preventDefault();

                        const $this = $(this);
                        const isExpanded = $this.hasClass(config.expandedClass);
                        $this.toggleClass(config.expandedClass)
                            .attr(config.ariaExpanded, !isExpanded);
                        $detailsRow.toggleClass(config.expandedClass);
                        $details.toggleClass(config.expandedClass)
                            .attr(config.ariaHidden, isExpanded);
                        $label.trigger('woonutri:section-toggled', {
                            section: $this,
                            expanded: !isExpanded
                        });
                        if (!isExpanded && !isElementInViewport($this[0])) {
                            $this[0].scrollIntoView({
                                behavior: 'smooth',
                                block: 'nearest'
                            });
                        }
                    });
                    $header.on('mouseenter', function () {
                        $(this).addClass('woonutri-hover');
                    }).on('mouseleave', function () {
                        $(this).removeClass('woonutri-hover');
                    });
                    $header.on('focus', function () {
                        $(this).addClass('woonutri-focused');
                    }).on('blur', function () {
                        $(this).removeClass('woonutri-focused');
                    });
                }
            });
        }
        addExpandCollapseControls($label);
    }

    
    function addExpandCollapseControls($label) {
        const $expandableRows = $label.find('.woonutri-expandable');

        if ($expandableRows.length > 1) {
            const $controls = $(`
                <div class="woonutri-expand-controls" style="text-align: right; margin-bottom: 0.5rem; font-size: 0.85em;">
                    <button type="button" class="woonutri-expand-all" style="background: none; border: none; color: var(--woonutri-primary, #2a7ae2); cursor: pointer; text-decoration: underline; margin-right: 0.5rem;">
                        ${config.expandAllText || 'Expand All'}
                    </button>
                    <button type="button" class="woonutri-collapse-all" style="background: none; border: none; color: var(--woonutri-primary, #2a7ae2); cursor: pointer; text-decoration: underline;">
                        ${config.collapseAllText || 'Collapse All'}
                    </button>
                </div>
            `);

            $label.prepend($controls);
            $controls.find('.woonutri-expand-all').on('click', function () {
                $expandableRows.each(function () {
                    const $header = $(this);
                    const $detailsRow = $header.next('tr');
                    const $details = $detailsRow.find('.woonutri-details');

                    if (!$header.hasClass(config.expandedClass)) {
                        $header.addClass(config.expandedClass)
                            .attr(config.ariaExpanded, 'true');
                        $detailsRow.addClass(config.expandedClass);
                        $details.addClass(config.expandedClass)
                            .attr(config.ariaHidden, 'false');
                    }
                });

                $label.trigger('woonutri:all-expanded');
            });
            $controls.find('.woonutri-collapse-all').on('click', function () {
                $expandableRows.each(function () {
                    const $header = $(this);
                    const $detailsRow = $header.next('tr');
                    const $details = $detailsRow.find('.woonutri-details');

                    if ($header.hasClass(config.expandedClass)) {
                        $header.removeClass(config.expandedClass)
                            .attr(config.ariaExpanded, 'false');
                        $detailsRow.removeClass(config.expandedClass);
                        $details.removeClass(config.expandedClass)
                            .attr(config.ariaHidden, 'true');
                    }
                });

                $label.trigger('woonutri:all-collapsed');
            });
        }
    }

    
    function isElementInViewport(el) {
        const rect = el.getBoundingClientRect();
        return (
            rect.top >= 0 &&
            rect.left >= 0 &&
            rect.bottom <= (window.innerHeight || document.documentElement.clientHeight) &&
            rect.right <= (window.innerWidth || document.documentElement.clientWidth)
        );
    }

    
    function addHoverEffects($label) {

        $label.find('tr').not(`.${config.detailsClass}`).hover(
            function () {
                $(this).css('transition', 'background-color 0.2s ease');
            },
            function () {
                $(this).css('transition', 'background-color 0.5s ease');
            }
        );
        $label.find('tr').attr('tabindex', '0').on('focus blur', function (e) {
            if (e.type === 'focus') {
                $(this).css({
                    'outline': '2px solid rgba(42, 122, 226, 0.3)',
                    'outline-offset': '-2px'
                });
            } else {
                $(this).css('outline', 'none');
            }
        });
    }

    
    function revealLabel($label) {

        $label.addClass(config.animatedClass);
        const $rows = $label.find(config.rowSelector);
        $rows.each(function (i) {
            const $row = $(this);
            setTimeout(() => {
                $row.addClass(config.animatedClass);
            }, i * config.staggerDelay);
        });
    }

    
    function setupPrintHandler() {

        window.addEventListener('beforeprint', function () {
            $(`.${config.detailsClass}`).addClass(config.expandedClass);
            $(`.${config.expandableClass}`).addClass(config.expandedClass)
                .attr(config.ariaExpanded, 'true');
        });
    }
    $(document).ready(function () {
        init();
        setupPrintHandler();
        $(document).on('ajaxComplete', function (event, xhr, settings) {

            setTimeout(init, 100);
        });
    });
    window.WooNutriLabel = {
        refresh: init,
        expandAll: function () {
            $(`.${config.expandableClass}`).addClass(config.expandedClass)
                .attr(config.ariaExpanded, 'true');
            $(`.${config.detailsClass}`).addClass(config.expandedClass)
                .attr(config.ariaHidden, 'false');
        },
        collapseAll: function () {
            $(`.${config.expandableClass}`).removeClass(config.expandedClass)
                .attr(config.ariaExpanded, 'false');
            $(`.${config.detailsClass}`).removeClass(config.expandedClass)
                .attr(config.ariaHidden, 'true');
        },
        
        updateForVariation: function (variationId, variationData) {
            handleVariationChange(variationId, variationData);
        }
    };

    
    function handleVariationChange(variationId, variationData) {
        const $label = $(config.labelSelector);

        if (!$label.length) return;
        if (variationData && variationData.woonutri_has_data) {

            $label.addClass('woonutri-loading');
            updateNutritionValues($label, variationData.woonutri_data);
            setTimeout(() => {
                $label.removeClass('woonutri-loading');
            }, 300);
            $label.trigger('woonutri:variation-changed', {
                variationId: variationId,
                hasOverride: variationData.woonutri_has_override
            });
        }
    }

    
    function updateNutritionValues($label, nutritionData) {
        if (!nutritionData) return;
        const $table = $label.find('.woonutri-table');
        const nutrientMap = {
            'energy_kcal': 'Calories',
            'fat': 'Fat',
            'saturated_fat': 'Saturated Fat',
            'trans_fat': 'Trans Fat',
            'carbs': 'Carbohydrates',
            'sugar': 'Sugar',
            'fiber': 'Fiber',
            'protein': 'Protein',
            'salt': 'Salt',
            'sodium': 'Sodium'
        };
        for (const [key, value] of Object.entries(nutritionData)) {

            if (!value || typeof value === 'object') continue;
            $table.find('tr').each(function () {
                const $row = $(this);
                const $label = $row.find('td:first-child');
                const $value = $row.find('td:last-child');
                if ($label.length && $value.length) {
                    const labelText = $label.text().toLowerCase();
                    const mappedLabel = nutrientMap[key] ? nutrientMap[key].toLowerCase() : key.replace(/_/g, ' ');

                    if (labelText.includes(mappedLabel)) {

                        $value.addClass('woonutri-value-updating');
                        setTimeout(() => {

                            const currentText = $value.text();
                            const unitMatch = currentText.match(/[a-zA-Z]+$/);
                            const unit = unitMatch ? unitMatch[0] : '';
                            $value.text(value + (unit ? ' ' + unit : ''));
                            $value.removeClass('woonutri-value-updating');
                        }, 150);
                    }
                }
            });
        }
    }

    
    function initVariationListeners() {

        $(document.body).on('found_variation', '.variations_form', function (event, variation) {
            if (variation && variation.variation_id) {
                handleVariationChange(variation.variation_id, variation);
            }
        });
        $(document.body).on('reset_data', '.variations_form', function () {

            const $label = $(config.labelSelector);
            if ($label.length) {
                $label.trigger('woonutri:variation-reset');

                if (typeof WooNutriLabel !== 'undefined') {
                    WooNutriLabel.refresh();
                }
            }
        });
    }
    $(document).ready(function () {
        initVariationListeners();
        const variationStyles = `
            <style id="woonutri-variation-styles">
                .woonutri-value-updating {
                    opacity: 0.5;
                    transition: opacity 0.15s ease;
                }
                
                .woonutri-label.woonutri-loading {
                    opacity: 0.7;
                }
                
                .woonutri-label.woonutri-loading::after {
                    content: '';
                    position: absolute;
                    top: 50%;
                    left: 50%;
                    width: 24px;
                    height: 24px;
                    margin: -12px 0 0 -12px;
                    border: 2px solid rgba(0,0,0,0.1);
                    border-top-color: #2a7ae2;
                    border-radius: 50%;
                    animation: woonutri-spin 0.8s linear infinite;
                }
            </style>
        `;

        if (!$('#woonutri-variation-styles').length) {
            $('head').append(variationStyles);
        }
    });

})(jQuery);