<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class Woonutri_Admin_UI
{
    public static function init()
    {
        add_filter('woocommerce_product_data_tabs', [__CLASS__, 'add_nutrition_tab']);
        add_action('woocommerce_product_data_panels', [__CLASS__, 'nutrition_tab_content']);
        add_action('woocommerce_process_product_meta', [__CLASS__, 'save_nutrition_fields']);
        add_action('admin_enqueue_scripts', [__CLASS__, 'enqueue_admin_assets']);
    }

    public static function enqueue_admin_assets($hook)
    {
        global $post;

        if (($hook == 'post.php' || $hook == 'post-new.php') && $post && $post->post_type === 'product') {
            wp_enqueue_style(
                'woonutri-admin-settings',
                WOONUTRI_PLUGIN_URL . 'assets/css/admin-settings.css',
                ['woonutri-admin-core'],
                WOONUTRI_VERSION
            );

            wp_enqueue_style(
                'woonutri-admin-core',
                WOONUTRI_PLUGIN_URL . 'assets/css/admin-core.css',
                [],
                WOONUTRI_VERSION . '.4'
            );


            wp_enqueue_script(
                'woonutri-admin',
                WOONUTRI_PLUGIN_URL . 'assets/js/admin-product.js',
                ['jquery', 'jquery-ui-accordion'],
                WOONUTRI_VERSION . '.2',
                true
            );

            wp_localize_script('woonutri-admin', 'woonutri_admin', [
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('woonutri_admin'),
                'is_pro' => function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code(),
                'active_standard' => self::get_active_standard()
            ]);
        }
    }

    public static function add_nutrition_tab($tabs)
    {
        $tabs['woonutri_nutrition'] = array(
            'label' => __('Nutrition', 'smart-nutrition-labels'),
            'target' => 'woonutri_nutrition_data',
            'class' => array(),
            'priority' => 60,
        );
        return $tabs;
    }

    public static function nutrition_tab_content()
    {
        global $post;
        $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();

        echo '<div id="woonutri_nutrition_data" class="panel woocommerce_options_panel woonutri-admin-panel">';

        if (!$is_pro) {
            self::render_free_user_fields($post->ID);
            echo '</div>';
            return;
        }

        $all_fields = self::get_all_nutrition_fields();
        $profiles = get_post_meta($post->ID, '_woonutri_nutrition_profiles', true);
        if (!is_array($profiles) || empty($profiles)) {
            $individual_data = Woonutri_Meta_Handler::get_nutrition_data($post->ID);
            if (!empty($individual_data)) {
                $profiles = [$individual_data];
            } else {
                $profiles = [[]];
            }
        }

        echo '<div id="woonutri-profiles-accordion">';

        foreach ($profiles as $index => $profile_data) {
            self::render_profile_fields($index, $profile_data, $all_fields);
        }

        echo '</div>';

        echo '<div class="woonutri-add-profile-section" style="margin: 20px 0; padding: 16px; background: var(--wn-bg-alt, #f8f9fa); border: 2px dashed var(--wn-border, #ddd); border-radius: 8px; text-align: center;">';
        echo '<button type="button" id="woonutri-add-profile" class="button button-primary" style="padding: 8px 20px; font-size: 14px;">';
        echo '<span class="dashicons dashicons-plus-alt2" style="margin-right: 6px; font-size: 16px; line-height: 1.5;"></span>';
        echo __('Add Nutrition Profile', 'smart-nutrition-labels');
        echo '</button>';
        echo '<p class="description" style="margin-top: 10px; color: var(--wn-text-secondary, #666);">' . __('Add multiple profiles for different serving sizes or product variations.', 'smart-nutrition-labels') . '</p>';
        echo '</div>';

        $nutrition_data = Woonutri_Meta_Handler::get_nutrition_data($post->ID);
        self::render_allergen_selection($post->ID, $nutrition_data);

        self::render_nutriscore_settings($post->ID);

        echo '<div id="woonutri-profile-template" style="display: none;">';
        self::render_profile_fields('{{index}}', [], $all_fields);
        echo '</div>';

        echo '</div>';
    }

    private static function render_free_user_fields($post_id)
    {
        $nutrition_data = Woonutri_Meta_Handler::get_nutrition_data($post_id);

        $core_fields = [
            'serving_size' => ['label' => __('Serving Size', 'smart-nutrition-labels'), 'unit' => '', 'placeholder' => '100g or 1 cup (240ml)', 'type' => 'text'],
            'energy_kcal' => ['label' => __('Energy', 'smart-nutrition-labels'), 'unit' => 'kcal', 'placeholder' => '250', 'type' => 'number'],
        ];

        $macro_fields = [
            'fat' => ['label' => __('Total Fat', 'smart-nutrition-labels'), 'unit' => 'g', 'placeholder' => '15.5', 'type' => 'number'],
            'saturated_fat' => ['label' => __('Saturated Fat', 'smart-nutrition-labels'), 'unit' => 'g', 'placeholder' => '5.0', 'type' => 'number'],
            'trans_fat' => ['label' => __('Trans Fat', 'smart-nutrition-labels'), 'unit' => 'g', 'placeholder' => '0', 'type' => 'number'],
            'cholesterol' => ['label' => __('Cholesterol', 'smart-nutrition-labels'), 'unit' => 'mg', 'placeholder' => '30', 'type' => 'number'],
            'sodium' => ['label' => __('Sodium', 'smart-nutrition-labels'), 'unit' => 'mg', 'placeholder' => '450', 'type' => 'number'],
            'carbs' => ['label' => __('Carbohydrates', 'smart-nutrition-labels'), 'unit' => 'g', 'placeholder' => '45.0', 'type' => 'number'],
            'fiber' => ['label' => __('Fiber', 'smart-nutrition-labels'), 'unit' => 'g', 'placeholder' => '3.2', 'type' => 'number'],
            'sugar' => ['label' => __('Sugars', 'smart-nutrition-labels'), 'unit' => 'g', 'placeholder' => '12.5', 'type' => 'number'],
            'added_sugars' => ['label' => __('Added Sugars', 'smart-nutrition-labels'), 'unit' => 'g', 'placeholder' => '8.0', 'type' => 'number'],
            'protein' => ['label' => __('Protein', 'smart-nutrition-labels'), 'unit' => 'g', 'placeholder' => '8.5', 'type' => 'number'],
            'salt' => ['label' => __('Salt', 'smart-nutrition-labels'), 'unit' => 'g', 'placeholder' => '1.5', 'type' => 'number'],
        ];

        $micro_fields = [
            'vitamin_d' => ['label' => __('Vitamin D', 'smart-nutrition-labels'), 'unit' => 'mcg', 'placeholder' => '2.5', 'type' => 'number'],
            'calcium' => ['label' => __('Calcium', 'smart-nutrition-labels'), 'unit' => 'mg', 'placeholder' => '120', 'type' => 'number'],
            'iron' => ['label' => __('Iron', 'smart-nutrition-labels'), 'unit' => 'mg', 'placeholder' => '3.6', 'type' => 'number'],
            'potassium' => ['label' => __('Potassium', 'smart-nutrition-labels'), 'unit' => 'mg', 'placeholder' => '350', 'type' => 'number'],
        ];

        echo '<div class="woonutri-free-fields">';

        echo '<div class="woonutri-section">';
        echo '<h4 class="woonutri-section-title">' . __('Core Information', 'smart-nutrition-labels') . '</h4>';
        echo '<div class="woonutri-fields-grid">';
        foreach ($core_fields as $key => $field) {
            self::render_single_field($key, $field, $nutrition_data);
        }
        echo '</div></div>';

        echo '<div class="woonutri-section">';
        echo '<h4 class="woonutri-section-title">' . __('Macronutrients', 'smart-nutrition-labels') . '</h4>';
        echo '<div class="woonutri-fields-grid">';
        foreach ($macro_fields as $key => $field) {
            self::render_single_field($key, $field, $nutrition_data);
        }
        echo '</div></div>';

        echo '<div class="woonutri-section">';
        echo '<h4 class="woonutri-section-title">' . __('Vitamins & Minerals', 'smart-nutrition-labels') . '</h4>';
        echo '<div class="woonutri-fields-grid">';
        foreach ($micro_fields as $key => $field) {
            self::render_single_field($key, $field, $nutrition_data);
        }
        echo '</div></div>';


        self::render_allergen_selection($post_id, $nutrition_data);

        echo '<div class="woonutri-pro-feature" style="margin-top: 24px;">';
        echo '<div class="woonutri-pro-active-badge" style="position: absolute; top: 16px; right: 20px;">PRO</div>';
        echo '<h3 style="margin: 0 0 12px 0; font-size: 16px;">' . __('Automate Your Workflow', 'smart-nutrition-labels') . '</h3>';
        echo '<p style="margin-bottom: 12px; color: var(--wn-text-light);">' . __('Save hours of manual data entry with our intelligent automation tools.', 'smart-nutrition-labels') . '</p>';
        echo '<ul style="column-count: 2; margin: 0 0 16px 0;">';
        echo '<li>' . __('Auto-calculate Daily Values') . '</li>';
        echo '<li>' . __('Smart Allergen Detection') . '</li>';
        echo '<li>' . __('Bulk Editing') . '</li>';
        echo '<li>' . __('Premium Label Themes') . '</li>';
        echo '</ul>';
        echo '<a href="' . esc_url(woonutri_fs()->get_upgrade_url()) . '" class="button button-primary">' . __('Upgrade to Pro', 'smart-nutrition-labels') . '</a>';
        echo '</div>';

        echo '</div>';
    }

    private static function render_single_field($key, $field, $data)
    {
        $value = isset($data[$key]) ? $data[$key] : '';
        $field_id = 'woonutri_' . $key;
        $field_label = $field['label'] . (!empty($field['unit']) ? ' <span class="woonutri-unit">(' . $field['unit'] . ')</span>' : '');

        echo '<div class="woonutri-field-wrapper">';
        woocommerce_wp_text_input([
            'id' => $field_id,
            'name' => $field_id,
            'label' => $field_label,
            'placeholder' => $field['placeholder'],
            'type' => $field['type'],
            'value' => $value,
            'custom_attributes' => ($field['type'] === 'number') ? ['step' => 'any', 'min' => '0'] : []
        ]);
        echo '</div>';
    }

    private static function render_allergen_selection($post_id, $nutrition_data = [])
    {
        $saved_allergens = isset($nutrition_data['allergens']) ? (array) $nutrition_data['allergens'] : [];

        if (is_string($saved_allergens)) {
            $saved_allergens = array_map('trim', explode(',', $saved_allergens));
        }

        $allergens = Woonutri_Allergens::get_eu_allergens();

        echo '<div class="woonutri-section woonutri-allergens-section">';
        echo '<h4 class="woonutri-section-title">';
        echo '<span style="margin-right: 8px;">Warning</span>' . __('Allergen Information', 'smart-nutrition-labels');
        echo '</h4>';
        echo '<p class="description" style="margin-bottom: 15px;">' . __('Select all allergens present in this product.', 'smart-nutrition-labels') . '</p>';

        echo '<div class="woonutri-allergen-grid">';

        foreach ($allergens as $key => $allergen) {
            $checked = in_array($key, $saved_allergens) || in_array($allergen['name'], $saved_allergens);
            $icon = isset($allergen['icon']) ? $allergen['icon'] : '';

            echo '<label class="woonutri-allergen-card ' . ($checked ? 'checked' : '') . '">';
            echo '<input type="checkbox" name="woonutri_allergens[]" value="' . esc_attr($key) . '"' . checked($checked, true, false) . '>';
            echo '<span class="woonutri-allergen-icon">' . esc_html($icon) . '</span>';
            echo '<span class="woonutri-allergen-name">' . esc_html($allergen['name']) . '</span>';
            echo '</label>';
        }

        echo '</div>';

        $may_contain = isset($nutrition_data['may_contain']) ? $nutrition_data['may_contain'] : '';
        echo '<div style="margin-top: 15px;">';
        woocommerce_wp_text_input([
            'id' => 'woonutri_may_contain',
            'name' => 'woonutri_may_contain',
            'label' => __('May Contain (Cross-Contamination)', 'smart-nutrition-labels'),
            'placeholder' => esc_attr__('e.g., traces of nuts, sesame', 'smart-nutrition-labels'),
            'value' => $may_contain,
            'desc_tip' => 'true',
            'description' => __('List any allergens that may be present due to cross-contamination.', 'smart-nutrition-labels')
        ]);
        echo '</div>';

        echo '</div>';

        self::render_dietary_preferences($post_id, $nutrition_data);
    }

    private static function render_dietary_preferences($post_id, $nutrition_data = [])
    {
        $saved_preferences = isset($nutrition_data['dietary_preferences']) ? (array) $nutrition_data['dietary_preferences'] : [];

        if (is_string($saved_preferences)) {
            $saved_preferences = array_map('trim', explode(',', $saved_preferences));
        }

        $preferences = Woonutri_Allergens::get_dietary_preferences();

        echo '<div class="woonutri-section woonutri-dietary-section">';
        echo '<h4 class="woonutri-section-title">';
        echo '<span style="margin-right: 8px;">Dietary</span>' . __('Dietary Preferences & Certifications', 'smart-nutrition-labels');
        echo '</h4>';
        echo '<p class="description" style="margin-bottom: 15px;">' . __('Select dietary claims and certifications that apply to this product.', 'smart-nutrition-labels') . '</p>';

        echo '<div class="woonutri-dietary-grid">';

        foreach ($preferences as $key => $preference) {
            $checked = in_array($key, $saved_preferences);
            $icon = isset($preference['icon']) ? $preference['icon'] : '';
            $type = isset($preference['type']) ? $preference['type'] : 'positive';
            $card_class = 'woonutri-dietary-card ' . ($checked ? 'checked' : '') . ' type-' . $type;

            echo '<label class="' . esc_attr($card_class) . '" title="' . esc_attr($preference['description']) . '">';
            echo '<input type="checkbox" name="woonutri_dietary[]" value="' . esc_attr($key) . '"' . checked($checked, true, false) . '>';
            echo '<span class="woonutri-dietary-icon">' . esc_html($icon) . '</span>';
            echo '<span class="woonutri-dietary-name">' . esc_html($preference['name']) . '</span>';
            echo '</label>';
        }

        echo '</div>';
        echo '</div>';
    }

    private static function get_all_nutrition_fields()
    {
        $basic_fields = [
            'energy_kcal' => ['label' => __('Energy', 'smart-nutrition-labels'), 'unit' => 'kcal', 'placeholder' => '250', 'type' => 'number'],
            'fat' => ['label' => __('Fat', 'smart-nutrition-labels'), 'unit' => 'g', 'placeholder' => '15.5', 'type' => 'number'],
            'carbs' => ['label' => __('Carbohydrates', 'smart-nutrition-labels'), 'unit' => 'g', 'placeholder' => '45.0', 'type' => 'number'],
            'sugar' => ['label' => __('Sugar', 'smart-nutrition-labels'), 'unit' => 'g', 'placeholder' => '12.5', 'type' => 'number'],
            'protein' => ['label' => __('Protein', 'smart-nutrition-labels'), 'unit' => 'g', 'placeholder' => '8.5', 'type' => 'number'],
            'fiber' => ['label' => __('Fiber', 'smart-nutrition-labels'), 'unit' => 'g', 'placeholder' => '3.2', 'type' => 'number'],
            'salt' => ['label' => __('Salt', 'smart-nutrition-labels'), 'unit' => 'g', 'placeholder' => '1.5', 'type' => 'number'],
            'trans_fat' => ['label' => __('Trans Fat', 'smart-nutrition-labels'), 'unit' => 'g', 'placeholder' => '0.1', 'type' => 'number'],
            'cholesterol' => ['label' => __('Cholesterol', 'smart-nutrition-labels'), 'unit' => 'mg', 'placeholder' => '30', 'type' => 'number'],
            'sodium' => ['label' => __('Sodium', 'smart-nutrition-labels'), 'unit' => 'mg', 'placeholder' => '450', 'type' => 'number'],
            'potassium' => ['label' => __('Potassium', 'smart-nutrition-labels'), 'unit' => 'mg', 'placeholder' => '350', 'type' => 'number'],
            'added_sugars' => ['label' => __('Added Sugars', 'smart-nutrition-labels'), 'unit' => 'g', 'placeholder' => '8.0', 'type' => 'number'],
            'vitamin_d' => ['label' => __('Vitamin D', 'smart-nutrition-labels'), 'unit' => 'mcg', 'placeholder' => '2.5', 'type' => 'number'],
            'calcium' => ['label' => __('Calcium', 'smart-nutrition-labels'), 'unit' => 'mg', 'placeholder' => '120', 'type' => 'number'],
            'iron' => ['label' => __('Iron', 'smart-nutrition-labels'), 'unit' => 'mg', 'placeholder' => '3.6', 'type' => 'number'],
        ];

        $pro_fields = [
            'serving_size' => ['label' => __('Serving Size', 'smart-nutrition-labels'), 'unit' => '', 'placeholder' => '1 cup (240ml)', 'type' => 'text'],
            'servings_per_container' => ['label' => __('Servings Per Container', 'smart-nutrition-labels'), 'unit' => '', 'placeholder' => '8', 'type' => 'text'],
            'saturated_fat' => ['label' => __('Saturated Fat', 'smart-nutrition-labels'), 'unit' => 'g', 'placeholder' => '5.0', 'type' => 'number'],
            'monounsaturated_fat' => ['label' => __('Monounsaturated Fat', 'smart-nutrition-labels'), 'unit' => 'g', 'placeholder' => '3.0', 'type' => 'number'],
            'polyunsaturated_fat' => ['label' => __('Polyunsaturated Fat', 'smart-nutrition-labels'), 'unit' => 'g', 'placeholder' => '2.0', 'type' => 'number'],
            'omega_3' => ['label' => __('Omega-3', 'smart-nutrition-labels'), 'unit' => 'g', 'placeholder' => '0.5', 'type' => 'number'],
            'omega_6' => ['label' => __('Omega-6', 'smart-nutrition-labels'), 'unit' => 'g', 'placeholder' => '1.0', 'type' => 'number'],
            'vitamin_a' => ['label' => __('Vitamin A', 'smart-nutrition-labels'), 'unit' => 'mcg', 'placeholder' => '150', 'type' => 'number'],
            'vitamin_c' => ['label' => __('Vitamin C', 'smart-nutrition-labels'), 'unit' => 'mg', 'placeholder' => '30', 'type' => 'number'],
            'vitamin_e' => ['label' => __('Vitamin E', 'smart-nutrition-labels'), 'unit' => 'mg', 'placeholder' => '5.0', 'type' => 'number'],
            'vitamin_k' => ['label' => __('Vitamin K', 'smart-nutrition-labels'), 'unit' => 'mcg', 'placeholder' => '20', 'type' => 'number'],
            'vitamin_b1' => ['label' => __('Thiamin (B1)', 'smart-nutrition-labels'), 'unit' => 'mg', 'placeholder' => '0.3', 'type' => 'number'],
            'vitamin_b2' => ['label' => __('Riboflavin (B2)', 'smart-nutrition-labels'), 'unit' => 'mg', 'placeholder' => '0.4', 'type' => 'number'],
            'vitamin_b3' => ['label' => __('Niacin (B3)', 'smart-nutrition-labels'), 'unit' => 'mg', 'placeholder' => '4.0', 'type' => 'number'],
            'vitamin_b6' => ['label' => __('Vitamin B6', 'smart-nutrition-labels'), 'unit' => 'mg', 'placeholder' => '0.5', 'type' => 'number'],
            'vitamin_b12' => ['label' => __('Vitamin B12', 'smart-nutrition-labels'), 'unit' => 'mcg', 'placeholder' => '1.2', 'type' => 'number'],
            'folate' => ['label' => __('Folate', 'smart-nutrition-labels'), 'unit' => 'mcg', 'placeholder' => '60', 'type' => 'number'],
            'biotin' => ['label' => __('Biotin', 'smart-nutrition-labels'), 'unit' => 'mcg', 'placeholder' => '10', 'type' => 'number'],
            'pantothenic_acid' => ['label' => __('Pantothenic Acid', 'smart-nutrition-labels'), 'unit' => 'mg', 'placeholder' => '1.5', 'type' => 'number'],
            'phosphorus' => ['label' => __('Phosphorus', 'smart-nutrition-labels'), 'unit' => 'mg', 'placeholder' => '100', 'type' => 'number'],
            'magnesium' => ['label' => __('Magnesium', 'smart-nutrition-labels'), 'unit' => 'mg', 'placeholder' => '40', 'type' => 'number'],
            'zinc' => ['label' => __('Zinc', 'smart-nutrition-labels'), 'unit' => 'mg', 'placeholder' => '2.0', 'type' => 'number'],
            'selenium' => ['label' => __('Selenium', 'smart-nutrition-labels'), 'unit' => 'mcg', 'placeholder' => '10', 'type' => 'number'],
            'copper' => ['label' => __('Copper', 'smart-nutrition-labels'), 'unit' => 'mg', 'placeholder' => '0.2', 'type' => 'number'],
            'manganese' => ['label' => __('Manganese', 'smart-nutrition-labels'), 'unit' => 'mg', 'placeholder' => '0.5', 'type' => 'number'],
            'chromium' => ['label' => __('Chromium', 'smart-nutrition-labels'), 'unit' => 'mcg', 'placeholder' => '10', 'type' => 'number'],
            'molybdenum' => ['label' => __('Molybdenum', 'smart-nutrition-labels'), 'unit' => 'mcg', 'placeholder' => '15', 'type' => 'number'],
            'iodine' => ['label' => __('Iodine', 'smart-nutrition-labels'), 'unit' => 'mcg', 'placeholder' => '30', 'type' => 'number'],
            'caffeine' => ['label' => __('Caffeine', 'smart-nutrition-labels'), 'unit' => 'mg', 'placeholder' => '0', 'type' => 'number'],
            'alcohol' => ['label' => __('Alcohol', 'smart-nutrition-labels'), 'unit' => 'g', 'placeholder' => '0', 'type' => 'number'],
            'vitamins' => ['label' => __('Other Vitamins & Minerals', 'smart-nutrition-labels'), 'unit' => '', 'placeholder' => 'Vitamin A: 15%, ...', 'type' => 'textarea'],
            'ingredients' => ['label' => __('Ingredients', 'smart-nutrition-labels'), 'unit' => '', 'placeholder' => 'Whole wheat flour, ...', 'type' => 'textarea'],
            'allergens' => ['label' => __('Allergens', 'smart-nutrition-labels'), 'unit' => '', 'placeholder' => 'Contains: Wheat, Milk, Soy.', 'type' => 'text'],
            'barcode' => ['label' => __('Barcode', 'smart-nutrition-labels'), 'unit' => '', 'placeholder' => '8901030865278', 'type' => 'text'],
        ];

        return array_merge([
            'profile_name' => ['label' => __('Profile Name', 'smart-nutrition-labels'), 'unit' => '', 'placeholder' => 'e.g., Per 100g', 'type' => 'text']
        ], $basic_fields, $pro_fields);
    }

    private static function render_profile_fields($index, $profile_data, $all_fields)
    {
        $profile_name = isset($profile_data['profile_name']) ? esc_attr($profile_data['profile_name']) : 'New Profile';
        ?>
        <div class="woonutri-profile-group woonutri-card" style="margin-bottom: 20px;">
            <h3 class="woonutri-accordion-header"
                style="display: flex; align-items: center; justify-content: space-between; margin: 0 0 16px 0; padding-bottom: 12px; border-bottom: 1px solid var(--wn-border);">
                <span class="woonutri-profile-name-text"
                    style="font-size: 16px; font-weight: 600; color: var(--wn-text-main);"><?php echo $profile_name; ?></span>
                <button type="button" class="button woonutri-remove-profile"
                    style="color: var(--wn-danger-text); border-color: var(--wn-danger-light);"><?php _e('Remove Profile', 'smart-nutrition-labels'); ?></button>
            </h3>
            <div class="woonutri-accordion-content">
                <?php
                $name_field = $all_fields['profile_name'];
                $field_id = 'woonutri_profiles[' . $index . '][profile_name]';
                echo '<div class="woonutri-section" style="margin-bottom: 20px;">';
                echo '<div class="woonutri-field-wrapper full-width">';
                woocommerce_wp_text_input([
                    'id' => $field_id,
                    'name' => $field_id,
                    'label' => $name_field['label'],
                    'placeholder' => $name_field['placeholder'],
                    'value' => isset($profile_data['profile_name']) ? $profile_data['profile_name'] : '',
                    'class' => 'woonutri-input'
                ]);
                echo '</div></div>';

                $sections = [
                    'core' => [
                        'title' => __('Basic Info', 'smart-nutrition-labels'),
                        'fields' => ['serving_size', 'servings_per_container', 'barcode']
                    ],
                    'macros' => [
                        'title' => __('Key Nutrients', 'smart-nutrition-labels'),
                        'fields' => ['energy_kcal', 'fat', 'saturated_fat', 'monounsaturated_fat', 'polyunsaturated_fat', 'trans_fat', 'omega_3', 'omega_6', 'cholesterol', 'sodium', 'carbs', 'fiber', 'sugar', 'added_sugars', 'protein', 'salt']
                    ],
                    'vitamins' => [
                        'title' => __('Vitamins', 'smart-nutrition-labels'),
                        'fields' => ['vitamin_a', 'vitamin_c', 'vitamin_d', 'vitamin_e', 'vitamin_k', 'vitamin_b1', 'vitamin_b2', 'vitamin_b3', 'vitamin_b6', 'vitamin_b12', 'folate', 'biotin', 'pantothenic_acid']
                    ],
                    'minerals' => [
                        'title' => __('Minerals', 'smart-nutrition-labels'),
                        'fields' => ['calcium', 'iron', 'potassium', 'phosphorus', 'magnesium', 'zinc', 'selenium', 'copper', 'manganese', 'chromium', 'molybdenum', 'iodine']
                    ],
                    'other' => [
                        'title' => __('Other', 'smart-nutrition-labels'),
                        'fields' => ['caffeine', 'alcohol']
                    ],
                    'lists' => [
                        'title' => __('Lists & Ingredients', 'smart-nutrition-labels'),
                        'fields' => ['vitamins', 'ingredients', 'allergens']
                    ]
                ];

                foreach ($sections as $section_key => $section) {
                    echo '<div class="woonutri-section" style="margin-top: 24px;">';
                    echo '<h4 class="woonutri-section-title" style="font-size: 13px; font-weight: 600; text-transform: uppercase; color: var(--wn-text-light); letter-spacing: 0.5px; border-bottom: 2px solid var(--wn-border); padding-bottom: 8px; margin-bottom: 16px;">' . $section['title'] . '</h4>';

                    $grid_class = ($section_key === 'lists') ? '' : 'woonutri-fields-grid';
                    echo '<div class="' . $grid_class . '">';

                    foreach ($section['fields'] as $key) {
                        if (!isset($all_fields[$key]))
                            continue;

                        $field = $all_fields[$key];
                        $value = isset($profile_data[$key]) ? $profile_data[$key] : '';
                        $field_id = 'woonutri_profiles[' . $index . '][' . $key . ']';
                        $field_label = $field['label'] . (!empty($field['unit']) ? ' <span class="woonutri-unit">(' . $field['unit'] . ')</span>' : '');

                        echo '<div class="woonutri-field-wrapper ' . ($field['type'] === 'textarea' ? 'full-width' : '') . '" style="margin-bottom: 12px;">';

                        if ($field['type'] === 'textarea') {
                            woocommerce_wp_textarea_input([
                                'id' => $field_id,
                                'name' => $field_id,
                                'label' => $field_label,
                                'placeholder' => $field['placeholder'],
                                'value' => $value,
                                'class' => 'woonutri-input',
                                'style' => 'height: 80px;'
                            ]);
                        } else {
                            woocommerce_wp_text_input([
                                'id' => $field_id,
                                'name' => $field_id,
                                'label' => $field_label,
                                'placeholder' => $field['placeholder'],
                                'type' => $field['type'],
                                'value' => $value,
                                'class' => 'woonutri-input',
                                'custom_attributes' => ($field['type'] === 'number') ? ['step' => 'any', 'min' => '0'] : []
                            ]);
                        }
                        echo '</div>';
                    }
                    echo '</div></div>';
                }
                ?>
            </div>
        </div>
        <?php
    }

    public static function save_nutrition_fields($post_id)
    {
        $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();

        if (!$is_pro) {
            $basic_fields = [
                'serving_size',
                'energy_kcal',
                'fat',
                'saturated_fat',
                'trans_fat',
                'cholesterol',
                'sodium',
                'carbs',
                'fiber',
                'sugar',
                'added_sugars',
                'protein',
                'salt',
                'vitamin_d',
                'calcium',
                'iron',
                'potassium'
            ];

            $nutrition_data = [];
            foreach ($basic_fields as $key) {
                $field_name = 'woonutri_' . $key;
                if (isset($_POST[$field_name])) {
                    $nutrition_data[$key] = sanitize_text_field($_POST[$field_name]);
                }
            }

            if (isset($_POST['woonutri_allergens']) && is_array($_POST['woonutri_allergens'])) {
                $nutrition_data['allergens'] = array_map('sanitize_text_field', $_POST['woonutri_allergens']);
            } else {
                $nutrition_data['allergens'] = [];
            }

            if (isset($_POST['woonutri_may_contain'])) {
                $nutrition_data['may_contain'] = sanitize_text_field($_POST['woonutri_may_contain']);
            }

            if (isset($_POST['woonutri_dietary']) && is_array($_POST['woonutri_dietary'])) {
                $nutrition_data['dietary_preferences'] = array_map('sanitize_text_field', $_POST['woonutri_dietary']);
            } else {
                $nutrition_data['dietary_preferences'] = [];
            }

            if (!empty($nutrition_data)) {
                Woonutri_Meta_Handler::set_nutrition_data($post_id, $nutrition_data);
            }

            if (class_exists('Woonutri_Cache')) {
                Woonutri_Cache::invalidate_dashboard();
            }
            return;
        }

        if (!isset($_POST['woonutri_profiles']) || !is_array($_POST['woonutri_profiles'])) {
            delete_post_meta($post_id, '_woonutri_nutrition_profiles');
            return;
        }

        $all_fields = self::get_all_nutrition_fields();
        $sanitized_profiles = [];

        foreach ($_POST['woonutri_profiles'] as $profile_data) {
            if (!is_array($profile_data))
                continue;

            $sanitized_profile = [];
            foreach ($all_fields as $key => $field) {
                if (isset($profile_data[$key])) {
                    if ($field['type'] === 'textarea') {
                        $sanitized_profile[$key] = wp_kses_post($profile_data[$key]);
                    } else {
                        $sanitized_profile[$key] = sanitize_text_field($profile_data[$key]);
                    }
                }
            }
            if (!empty($sanitized_profile)) {
                $sanitized_profiles[] = $sanitized_profile;
            }
        }

        if (!empty($sanitized_profiles)) {
            update_post_meta($post_id, '_woonutri_nutrition_profiles', $sanitized_profiles);

            $first_profile = $sanitized_profiles[0];

            if (isset($_POST['woonutri_allergens']) && is_array($_POST['woonutri_allergens'])) {
                $first_profile['allergens'] = array_map('sanitize_text_field', $_POST['woonutri_allergens']);
            } else {
                $first_profile['allergens'] = [];
            }

            if (isset($_POST['woonutri_may_contain'])) {
                $first_profile['may_contain'] = sanitize_text_field($_POST['woonutri_may_contain']);
            }

            if (isset($_POST['woonutri_dietary']) && is_array($_POST['woonutri_dietary'])) {
                $first_profile['dietary_preferences'] = array_map('sanitize_text_field', $_POST['woonutri_dietary']);
            } else {
                $first_profile['dietary_preferences'] = [];
            }

            Woonutri_Meta_Handler::set_nutrition_data($post_id, $first_profile);
        } else {
            delete_post_meta($post_id, '_woonutri_nutrition_profiles');

            $extra_data = [];
            if (isset($_POST['woonutri_allergens']) && is_array($_POST['woonutri_allergens'])) {
                $extra_data['allergens'] = array_map('sanitize_text_field', $_POST['woonutri_allergens']);
            }
            if (isset($_POST['woonutri_may_contain'])) {
                $extra_data['may_contain'] = sanitize_text_field($_POST['woonutri_may_contain']);
            }
            if (isset($_POST['woonutri_dietary']) && is_array($_POST['woonutri_dietary'])) {
                $extra_data['dietary_preferences'] = array_map('sanitize_text_field', $_POST['woonutri_dietary']);
            }
            if (!empty($extra_data)) {
                Woonutri_Meta_Handler::set_nutrition_data($post_id, $extra_data);
            }
        }

        if (isset($_POST['woonutri_nutriscore_enabled'])) {
            $allowed_enabled = ['inherit', 'yes', 'no'];
            $nutriscore_enabled = sanitize_text_field($_POST['woonutri_nutriscore_enabled']);
            if (in_array($nutriscore_enabled, $allowed_enabled)) {
                update_post_meta($post_id, '_woonutri_nutriscore_enabled', $nutriscore_enabled);
            }
        }

        if (isset($_POST['woonutri_nutriscore_override'])) {
            $allowed_grades = ['auto', 'A', 'B', 'C', 'D', 'E'];
            $nutriscore_override = sanitize_text_field($_POST['woonutri_nutriscore_override']);
            if (in_array($nutriscore_override, $allowed_grades)) {
                update_post_meta($post_id, '_woonutri_nutriscore_override', $nutriscore_override);
            }
        }

        if (isset($_POST['woonutri_product_type'])) {
            $allowed_types = ['food', 'beverage'];
            $product_type = sanitize_text_field($_POST['woonutri_product_type']);
            if (in_array($product_type, $allowed_types)) {
                update_post_meta($post_id, '_woonutri_product_type', $product_type);
            }
        }

        if (class_exists('Woonutri_Cache')) {
            Woonutri_Cache::invalidate_dashboard();
        }
    }

    private static function get_active_standard()
    {
        $options = get_option('woonutri_options', []);
        $theme = isset($options['label_theme']) ? $options['label_theme'] : 'basic';

        if (in_array($theme, ['fda', 'us-fda'])) {
            return 'fda';
        } elseif (in_array($theme, ['eu', 'uk', 'minimal'])) {
            return 'eu';
        }

        return 'basic';
    }

    private static function render_nutriscore_settings($post_id)
    {
        $nutriscore_enabled = get_post_meta($post_id, '_woonutri_nutriscore_enabled', true);
        $nutriscore_override = get_post_meta($post_id, '_woonutri_nutriscore_override', true);
        $product_type = get_post_meta($post_id, '_woonutri_product_type', true);

        if ($nutriscore_enabled === '') {
            $nutriscore_enabled = 'inherit';
        }
        if ($nutriscore_override === '') {
            $nutriscore_override = 'auto';
        }
        if ($product_type === '') {
            $product_type = 'food';
        }

        echo '<div class="woonutri-section woonutri-nutriscore-settings">';
        echo '<h4 class="woonutri-section-title">';
        echo '<span style="margin-right: 8px;">Label</span>' . __('Nutri-Score Settings', 'smart-nutrition-labels');
        echo '</h4>';
        echo '<p class="description" style="margin-bottom: 15px;">' . __('Control how Nutri-Score is displayed for this product.', 'smart-nutrition-labels') . '</p>';

        echo '<div style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 20px;">';

        echo '<div style="display: flex; flex-direction: column;">';
        echo '<label style="font-weight: 600; margin-bottom: 8px; display: block; font-size: 13px; color: #1e293b;">' . __('Display Nutri-Score', 'smart-nutrition-labels') . '</label>';
        echo '<select name="woonutri_nutriscore_enabled" style="width: 100%; border: 1px solid #e2e8f0; border-radius: 8px; padding: 8px 12px; font-size: 14px; min-height: 38px;">';
        echo '<option value="inherit"' . selected($nutriscore_enabled, 'inherit', false) . '>' . __('Use Global Setting', 'smart-nutrition-labels') . '</option>';
        echo '<option value="yes"' . selected($nutriscore_enabled, 'yes', false) . '>' . __('Show Nutri-Score', 'smart-nutrition-labels') . '</option>';
        echo '<option value="no"' . selected($nutriscore_enabled, 'no', false) . '>' . __('Hide Nutri-Score', 'smart-nutrition-labels') . '</option>';
        echo '</select>';
        echo '<p class="description" style="font-size: 12px; color: #64748b; margin-top: 6px;">' . __('Override the global Nutri-Score display setting for this product.', 'smart-nutrition-labels') . '</p>';
        echo '</div>';

        echo '<div style="display: flex; flex-direction: column;">';
        echo '<label style="font-weight: 600; margin-bottom: 8px; display: block; font-size: 13px; color: #1e293b;">' . __('Nutri-Score Grade', 'smart-nutrition-labels') . '</label>';
        echo '<select name="woonutri_nutriscore_override" style="width: 100%; border: 1px solid #e2e8f0; border-radius: 8px; padding: 8px 12px; font-size: 14px; min-height: 38px;">';
        echo '<option value="auto"' . selected($nutriscore_override, 'auto', false) . '>' . __('Calculate Automatically', 'smart-nutrition-labels') . '</option>';
        echo '<option value="A"' . selected($nutriscore_override, 'A', false) . '>A - ' . __('Excellent', 'smart-nutrition-labels') . '</option>';
        echo '<option value="B"' . selected($nutriscore_override, 'B', false) . '>B - ' . __('Good', 'smart-nutrition-labels') . '</option>';
        echo '<option value="C"' . selected($nutriscore_override, 'C', false) . '>C - ' . __('Average', 'smart-nutrition-labels') . '</option>';
        echo '<option value="D"' . selected($nutriscore_override, 'D', false) . '>D - ' . __('Poor', 'smart-nutrition-labels') . '</option>';
        echo '<option value="E"' . selected($nutriscore_override, 'E', false) . '>E - ' . __('Bad', 'smart-nutrition-labels') . '</option>';
        echo '</select>';
        echo '<p class="description" style="font-size: 12px; color: #64748b; margin-top: 6px;">' . __('Manually set the Nutri-Score grade instead of calculating from nutrition data.', 'smart-nutrition-labels') . '</p>';
        echo '</div>';

        echo '<div style="display: flex; flex-direction: column;">';
        echo '<label style="font-weight: 600; margin-bottom: 8px; display: block; font-size: 13px; color: #1e293b;">' . __('Product Type', 'smart-nutrition-labels') . '</label>';
        echo '<select name="woonutri_product_type" style="width: 100%; border: 1px solid #e2e8f0; border-radius: 8px; padding: 8px 12px; font-size: 14px; min-height: 38px;">';
        echo '<option value="food"' . selected($product_type, 'food', false) . '>' . __('Food', 'smart-nutrition-labels') . '</option>';
        echo '<option value="beverage"' . selected($product_type, 'beverage', false) . '>' . __('Beverage', 'smart-nutrition-labels') . '</option>';
        echo '</select>';
        echo '<p class="description" style="font-size: 12px; color: #64748b; margin-top: 6px;">' . __('Beverages use different Nutri-Score thresholds (stricter on sugar).', 'smart-nutrition-labels') . '</p>';
        echo '</div>';

        echo '</div>';
        echo '</div>';
    }
}
