<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Woonutri_Setup_Wizard {

    public static function init() {
        add_action( 'admin_menu', [ __CLASS__, 'add_wizard_page' ] );
        add_action( 'admin_init', [ __CLASS__, 'maybe_redirect_to_wizard' ] );
        add_action( 'admin_enqueue_scripts', [ __CLASS__, 'enqueue_wizard_assets' ] );
        add_action( 'wp_ajax_woonutri_save_wizard_step', [ __CLASS__, 'ajax_save_step' ] );
        add_action( 'wp_ajax_woonutri_complete_wizard', [ __CLASS__, 'ajax_complete_wizard' ] );
        add_action( 'wp_ajax_woonutri_skip_wizard', [ __CLASS__, 'ajax_skip_wizard' ] );
    }

    public static function add_wizard_page() {
        add_submenu_page(
            null,
            __( 'WooNutri Setup', 'woonutri' ),
            __( 'Setup Wizard', 'woonutri' ),
            'manage_woocommerce',
            'woonutri-setup',
            [ __CLASS__, 'render_wizard_page' ]
        );
    }

    public static function maybe_redirect_to_wizard() {
        if ( get_transient( 'woonutri_activation_redirect' ) ) {
            delete_transient( 'woonutri_activation_redirect' );

            if ( get_option( 'woonutri_wizard_completed' ) ) {
                return;
            }

            if ( isset( $_GET['activate-multi'] ) ) {
                return;
            }

            wp_safe_redirect( admin_url( 'admin.php?page=woonutri-setup' ) );
            exit;
        }
    }

    public static function enqueue_wizard_assets( $hook ) {
        if ( $hook !== 'admin_page_woonutri-setup' ) {
            return;
        }

        wp_enqueue_style(
            'woonutri-setup-wizard',
            WOONUTRI_PLUGIN_URL . 'assets/css/setup-wizard.css',
            [],
            WOONUTRI_VERSION
        );

        wp_enqueue_script(
            'woonutri-setup-wizard',
            WOONUTRI_PLUGIN_URL . 'assets/js/setup-wizard.js',
            [ 'jquery' ],
            WOONUTRI_VERSION,
            true
        );

        wp_localize_script( 'woonutri-setup-wizard', 'woonutriWizard', [
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'nonce' => wp_create_nonce( 'woonutri_wizard' ),
            'settings_url' => admin_url( 'admin.php?page=woonutri-settings' ),
            'dashboard_url' => admin_url( 'admin.php?page=woonutri-dashboard' ),
            'i18n' => [
                'saving' => __( 'Saving...', 'woonutri' ),
                'next' => __( 'Next Step', 'woonutri' ),
                'complete' => __( 'Complete Setup', 'woonutri' ),
                'error' => __( 'An error occurred. Please try again.', 'woonutri' ),
            ]
        ] );
    }

    public static function render_wizard_page() {
        $current_step = isset( $_GET['step'] ) ? absint( $_GET['step'] ) : 1;
        $total_steps = 4;
        ?>
        <div class="woonutri-wizard-wrap">
            <div class="woonutri-wizard-container">
                <div class="woonutri-wizard-header">
                    <div class="woonutri-wizard-logo">
                        <span class="woonutri-logo-icon">🥗</span>
                        <h1><?php _e( 'WooNutri Setup', 'woonutri' ); ?></h1>
                    </div>
                    <p class="woonutri-wizard-subtitle">
                        <?php _e( 'Let\'s configure your nutrition labels in just a few steps.', 'woonutri' ); ?>
                    </p>
                </div>

                <div class="woonutri-wizard-progress">
                    <div class="woonutri-progress-bar">
                        <div class="woonutri-progress-fill" style="width: <?php echo ( $current_step / $total_steps ) * 100; ?>%"></div>
                    </div>
                    <div class="woonutri-progress-steps">
                        <?php for ( $i = 1; $i <= $total_steps; $i++ ) : ?>
                            <div class="woonutri-progress-step <?php echo $i <= $current_step ? 'active' : ''; ?> <?php echo $i < $current_step ? 'completed' : ''; ?>">
                                <span class="step-number"><?php echo $i < $current_step ? '✓' : $i; ?></span>
                                <span class="step-label"><?php echo self::get_step_label( $i ); ?></span>
                            </div>
                        <?php endfor; ?>
                    </div>
                </div>

                <div class="woonutri-wizard-content" data-step="<?php echo $current_step; ?>">
                    <?php self::render_step( $current_step ); ?>
                </div>

                <div class="woonutri-wizard-footer">
                    <div class="woonutri-wizard-actions">
                        <?php if ( $current_step > 1 ) : ?>
                            <a href="<?php echo admin_url( 'admin.php?page=woonutri-setup&step=' . ( $current_step - 1 ) ); ?>" class="button woonutri-wizard-prev">
                                ← <?php _e( 'Previous', 'woonutri' ); ?>
                            </a>
                        <?php else : ?>
                            <button type="button" class="button woonutri-wizard-skip" data-action="skip">
                                <?php _e( 'Skip Setup', 'woonutri' ); ?>
                            </button>
                        <?php endif; ?>

                        <?php if ( $current_step < $total_steps ) : ?>
                            <button type="button" class="button button-primary woonutri-wizard-next" data-step="<?php echo $current_step; ?>">
                                <?php _e( 'Next Step', 'woonutri' ); ?> →
                            </button>
                        <?php else : ?>
                            <button type="button" class="button button-primary woonutri-wizard-complete" data-step="<?php echo $current_step; ?>">
                                <?php _e( 'Complete Setup', 'woonutri' ); ?> ✓
                            </button>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    private static function get_step_label( $step ) {
        $labels = [
            1 => __( 'Region', 'woonutri' ),
            2 => __( 'Display', 'woonutri' ),
            3 => __( 'Style', 'woonutri' ),
            4 => __( 'Finish', 'woonutri' ),
        ];
        return isset( $labels[ $step ] ) ? $labels[ $step ] : '';
    }

    private static function render_step( $step ) {
        switch ( $step ) {
            case 1:
                self::render_step_region();
                break;
            case 2:
                self::render_step_display();
                break;
            case 3:
                self::render_step_style();
                break;
            case 4:
                self::render_step_finish();
                break;
        }
    }

    private static function render_step_region() {
        $options = get_option( 'woonutri_options', [] );
        $region = isset( $options['region'] ) ? $options['region'] : 'eu';
        ?>
        <div class="woonutri-step-content">
            <h2><?php _e( 'Select Your Region', 'woonutri' ); ?></h2>
            <p class="description"><?php _e( 'Choose your region to apply the correct nutrition label format and compliance standards.', 'woonutri' ); ?></p>

            <div class="woonutri-region-cards">
                <label class="woonutri-region-card <?php echo $region === 'eu' ? 'selected' : ''; ?>">
                    <input type="radio" name="woonutri_region" value="eu" <?php checked( $region, 'eu' ); ?>>
                    <div class="region-icon">🇪🇺</div>
                    <div class="region-name"><?php _e( 'European Union', 'woonutri' ); ?></div>
                    <div class="region-desc"><?php _e( 'EU 1169/2011 compliant labels with per 100g format', 'woonutri' ); ?></div>
                    <div class="region-features">
                        <span>✓ <?php _e( 'Per 100g/100ml', 'woonutri' ); ?></span>
                        <span>✓ <?php _e( 'Reference Intake %', 'woonutri' ); ?></span>
                        <span>✓ <?php _e( 'kJ & kcal', 'woonutri' ); ?></span>
                    </div>
                </label>

                <label class="woonutri-region-card <?php echo $region === 'us' ? 'selected' : ''; ?>">
                    <input type="radio" name="woonutri_region" value="us" <?php checked( $region, 'us' ); ?>>
                    <div class="region-icon">🇺🇸</div>
                    <div class="region-name"><?php _e( 'United States', 'woonutri' ); ?></div>
                    <div class="region-desc"><?php _e( 'FDA compliant Nutrition Facts labels', 'woonutri' ); ?></div>
                    <div class="region-features">
                        <span>✓ <?php _e( 'Per Serving', 'woonutri' ); ?></span>
                        <span>✓ <?php _e( '% Daily Value', 'woonutri' ); ?></span>
                        <span>✓ <?php _e( 'Calories', 'woonutri' ); ?></span>
                    </div>
                </label>

                <label class="woonutri-region-card <?php echo $region === 'uk' ? 'selected' : ''; ?>">
                    <input type="radio" name="woonutri_region" value="uk" <?php checked( $region, 'uk' ); ?>>
                    <div class="region-icon">🇬🇧</div>
                    <div class="region-name"><?php _e( 'United Kingdom', 'woonutri' ); ?></div>
                    <div class="region-desc"><?php _e( 'UK post-Brexit nutrition information format', 'woonutri' ); ?></div>
                    <div class="region-features">
                        <span>✓ <?php _e( 'Per 100g/Serving', 'woonutri' ); ?></span>
                        <span>✓ <?php _e( 'Reference Intake', 'woonutri' ); ?></span>
                        <span>✓ <?php _e( 'Traffic Lights', 'woonutri' ); ?></span>
                    </div>
                </label>

                <label class="woonutri-region-card <?php echo $region === 'other' ? 'selected' : ''; ?>">
                    <input type="radio" name="woonutri_region" value="other" <?php checked( $region, 'other' ); ?>>
                    <div class="region-icon">🌍</div>
                    <div class="region-name"><?php _e( 'Other / Custom', 'woonutri' ); ?></div>
                    <div class="region-desc"><?php _e( 'Use flexible settings for other regions', 'woonutri' ); ?></div>
                    <div class="region-features">
                        <span>✓ <?php _e( 'Customizable', 'woonutri' ); ?></span>
                        <span>✓ <?php _e( 'All formats', 'woonutri' ); ?></span>
                        <span>✓ <?php _e( 'Flexible units', 'woonutri' ); ?></span>
                    </div>
                </label>
            </div>
        </div>
        <?php
    }

    private static function render_step_display() {
        $options = get_option( 'woonutri_options', [] );
        $position = isset( $options['display_position'] ) ? $options['display_position'] : 'tab';
        ?>
        <div class="woonutri-step-content">
            <h2><?php _e( 'Choose Display Position', 'woonutri' ); ?></h2>
            <p class="description"><?php _e( 'Where should nutrition labels appear on your product pages?', 'woonutri' ); ?></p>

            <div class="woonutri-display-options">
                <label class="woonutri-display-option <?php echo $position === 'summary' ? 'selected' : ''; ?>">
                    <input type="radio" name="woonutri_display" value="summary" <?php checked( $position, 'summary' ); ?>>
                    <div class="display-preview">
                        <div class="preview-image preview-summary">
                            <div class="mock-product-image"></div>
                            <div class="mock-product-info">
                                <div class="mock-title"></div>
                                <div class="mock-price"></div>
                                <div class="mock-nutrition highlight"></div>
                            </div>
                        </div>
                    </div>
                    <div class="display-name"><?php _e( 'Product Summary', 'woonutri' ); ?></div>
                    <div class="display-desc"><?php _e( 'Show in the product summary area next to price and add to cart', 'woonutri' ); ?></div>
                </label>

                <label class="woonutri-display-option <?php echo $position === 'tab' ? 'selected' : ''; ?>">
                    <input type="radio" name="woonutri_display" value="tab" <?php checked( $position, 'tab' ); ?>>
                    <div class="display-preview">
                        <div class="preview-image preview-tab">
                            <div class="mock-tabs">
                                <span class="mock-tab">Description</span>
                                <span class="mock-tab highlight">Nutrition</span>
                                <span class="mock-tab">Reviews</span>
                            </div>
                            <div class="mock-tab-content"></div>
                        </div>
                    </div>
                    <div class="display-name"><?php _e( 'Product Tab', 'woonutri' ); ?></div>
                    <div class="display-desc"><?php _e( 'Add a dedicated "Nutrition Facts" tab to product tabs', 'woonutri' ); ?></div>
                </label>

                <label class="woonutri-display-option <?php echo $position === 'both' ? 'selected' : ''; ?>">
                    <input type="radio" name="woonutri_display" value="both" <?php checked( $position, 'both' ); ?>>
                    <div class="display-preview">
                        <div class="preview-image preview-both">
                            <div class="mock-product-image small"></div>
                            <div class="mock-product-info small">
                                <div class="mock-nutrition highlight small"></div>
                            </div>
                            <div class="mock-tabs small">
                                <span class="mock-tab highlight">Nutrition</span>
                            </div>
                        </div>
                    </div>
                    <div class="display-name"><?php _e( 'Both Locations', 'woonutri' ); ?></div>
                    <div class="display-desc"><?php _e( 'Display in both summary and as a tab for maximum visibility', 'woonutri' ); ?></div>
                </label>
            </div>
        </div>
        <?php
    }

    private static function render_step_style() {
        $options = get_option( 'woonutri_options', [] );
        $theme = isset( $options['label_theme'] ) ? $options['label_theme'] : 'basic';
        $is_pro = function_exists( 'woonutri_fs' ) && woonutri_fs()->can_use_premium_code();
        ?>
        <div class="woonutri-step-content">
            <h2><?php _e( 'Choose Your Label Style', 'woonutri' ); ?></h2>
            <p class="description"><?php _e( 'Select a visual style for your nutrition labels.', 'woonutri' ); ?></p>

            <div class="woonutri-theme-grid">
                <label class="woonutri-theme-card <?php echo $theme === 'basic' ? 'selected' : ''; ?>">
                    <input type="radio" name="woonutri_theme" value="basic" <?php checked( $theme, 'basic' ); ?>>
                    <div class="theme-preview theme-basic">
                        <div class="mock-label">
                            <div class="mock-header">Nutrition Facts</div>
                            <div class="mock-row"></div>
                            <div class="mock-row"></div>
                            <div class="mock-row"></div>
                        </div>
                    </div>
                    <div class="theme-name"><?php _e( 'Basic', 'woonutri' ); ?></div>
                    <span class="theme-badge free"><?php _e( 'FREE', 'woonutri' ); ?></span>
                </label>

                <?php
                $pro_themes = [
                    'fda' => [ 'name' => 'FDA', 'desc' => 'US FDA Style' ],
                    'eu' => [ 'name' => 'EU Compliant', 'desc' => 'European Format' ],
                    'modern' => [ 'name' => 'Modern', 'desc' => 'Clean Design' ],
                    'dark' => [ 'name' => 'Dark', 'desc' => 'Dark Mode' ],
                    'premium' => [ 'name' => 'Premium', 'desc' => 'Gold Accents' ],
                ];

                foreach ( $pro_themes as $theme_key => $theme_data ) :
                    $disabled = ! $is_pro;
                ?>
                <label class="woonutri-theme-card <?php echo $theme === $theme_key ? 'selected' : ''; ?> <?php echo $disabled ? 'locked' : ''; ?>">
                    <input type="radio" name="woonutri_theme" value="<?php echo $theme_key; ?>" <?php checked( $theme, $theme_key ); ?> <?php echo $disabled ? 'disabled' : ''; ?>>
                    <div class="theme-preview theme-<?php echo $theme_key; ?>">
                        <div class="mock-label">
                            <div class="mock-header">Nutrition Facts</div>
                            <div class="mock-row"></div>
                            <div class="mock-row"></div>
                            <div class="mock-row"></div>
                        </div>
                        <?php if ( $disabled ) : ?>
                        <div class="lock-overlay">
                            <span class="lock-icon">🔒</span>
                        </div>
                        <?php endif; ?>
                    </div>
                    <div class="theme-name"><?php echo esc_html( $theme_data['name'] ); ?></div>
                    <span class="theme-badge pro"><?php _e( 'PRO', 'woonutri' ); ?></span>
                </label>
                <?php endforeach; ?>
            </div>

            <?php if ( ! $is_pro ) : ?>
            <div class="woonutri-upgrade-prompt">
                <div class="upgrade-icon">⭐</div>
                <div class="upgrade-text">
                    <strong><?php _e( 'Unlock Premium Themes', 'woonutri' ); ?></strong>
                    <p><?php _e( 'Get access to FDA, EU, Modern, Dark, and Premium themes with Pro.', 'woonutri' ); ?></p>
                </div>
                <a href="<?php echo esc_url( woonutri_fs()->get_upgrade_url() ); ?>" class="button button-primary">
                    <?php _e( 'Upgrade to Pro', 'woonutri' ); ?>
                </a>
            </div>
            <?php endif; ?>
        </div>
        <?php
    }

    private static function render_step_finish() {
        $options = get_option( 'woonutri_options', [] );
        $is_pro = function_exists( 'woonutri_fs' ) && woonutri_fs()->can_use_premium_code();
        ?>
        <div class="woonutri-step-content woonutri-step-finish">
            <div class="finish-icon">🎉</div>
            <h2><?php _e( 'You\'re All Set!', 'woonutri' ); ?></h2>
            <p class="description"><?php _e( 'WooNutri is now configured and ready to use. Here\'s a summary of your settings:', 'woonutri' ); ?></p>

            <div class="woonutri-settings-summary">
                <div class="summary-item">
                    <span class="summary-label"><?php _e( 'Region:', 'woonutri' ); ?></span>
                    <span class="summary-value"><?php echo self::get_region_name( $options['region'] ?? 'eu' ); ?></span>
                </div>
                <div class="summary-item">
                    <span class="summary-label"><?php _e( 'Display Position:', 'woonutri' ); ?></span>
                    <span class="summary-value"><?php echo self::get_display_name( $options['display_position'] ?? 'tab' ); ?></span>
                </div>
                <div class="summary-item">
                    <span class="summary-label"><?php _e( 'Label Theme:', 'woonutri' ); ?></span>
                    <span class="summary-value"><?php echo ucfirst( $options['label_theme'] ?? 'basic' ); ?></span>
                </div>
            </div>

            <div class="woonutri-next-steps">
                <h3><?php _e( 'Next Steps', 'woonutri' ); ?></h3>
                <div class="next-steps-grid">
                    <a href="<?php echo admin_url( 'edit.php?post_type=product' ); ?>" class="next-step-card">
                        <div class="step-icon">📦</div>
                        <div class="step-title"><?php _e( 'Add Nutrition Data', 'woonutri' ); ?></div>
                        <div class="step-desc"><?php _e( 'Edit a product and go to the Nutrition tab', 'woonutri' ); ?></div>
                    </a>

                    <a href="<?php echo admin_url( 'admin.php?page=woonutri-settings' ); ?>" class="next-step-card">
                        <div class="step-icon">⚙️</div>
                        <div class="step-title"><?php _e( 'Customize Settings', 'woonutri' ); ?></div>
                        <div class="step-desc"><?php _e( 'Fine-tune colors, fonts, and layout', 'woonutri' ); ?></div>
                    </a>

                    <?php if ( ! $is_pro ) : ?>
                    <a href="<?php echo esc_url( woonutri_fs()->get_upgrade_url() ); ?>" class="next-step-card pro-card">
                        <div class="step-icon">⭐</div>
                        <div class="step-title"><?php _e( 'Upgrade to Pro', 'woonutri' ); ?></div>
                        <div class="step-desc"><?php _e( 'Unlock AI scanning, FDA themes & more', 'woonutri' ); ?></div>
                    </a>
                    <?php else : ?>
                    <a href="<?php echo admin_url( 'admin.php?page=woonutri-dashboard' ); ?>" class="next-step-card">
                        <div class="step-icon">📊</div>
                        <div class="step-title"><?php _e( 'View Dashboard', 'woonutri' ); ?></div>
                        <div class="step-desc"><?php _e( 'Check analytics and compliance status', 'woonutri' ); ?></div>
                    </a>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php
    }

    private static function get_region_name( $region ) {
        $regions = [
            'eu' => __( 'European Union', 'woonutri' ),
            'us' => __( 'United States (FDA)', 'woonutri' ),
            'uk' => __( 'United Kingdom', 'woonutri' ),
            'other' => __( 'Other / Custom', 'woonutri' ),
        ];
        return isset( $regions[ $region ] ) ? $regions[ $region ] : $region;
    }

    private static function get_display_name( $position ) {
        $positions = [
            'summary' => __( 'Product Summary', 'woonutri' ),
            'tab' => __( 'Product Tab', 'woonutri' ),
            'both' => __( 'Both Locations', 'woonutri' ),
        ];
        return isset( $positions[ $position ] ) ? $positions[ $position ] : $position;
    }

    public static function ajax_save_step() {
        check_ajax_referer( 'woonutri_wizard', 'nonce' );

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error( [ 'message' => __( 'Permission denied', 'woonutri' ) ] );
        }

        $step = isset( $_POST['step'] ) ? absint( $_POST['step'] ) : 1;
        $options = get_option( 'woonutri_options', [] );

        switch ( $step ) {
            case 1:
                $region = isset( $_POST['region'] ) ? sanitize_text_field( $_POST['region'] ) : 'eu';
                $options['region'] = $region;

                if ( $region === 'us' ) {
                    $options['label_theme'] = 'basic';
                    $options['display_format'] = 'per_serving';
                } else {
                    $options['display_format'] = 'per_100g';
                }
                break;

            case 2:
                $display = isset( $_POST['display'] ) ? sanitize_text_field( $_POST['display'] ) : 'tab';
                $options['display_position'] = $display;
                break;

            case 3:
                $theme = isset( $_POST['theme'] ) ? sanitize_text_field( $_POST['theme'] ) : 'basic';
                $is_pro = function_exists( 'woonutri_fs' ) && woonutri_fs()->can_use_premium_code();

                if ( ! $is_pro && $theme !== 'basic' ) {
                    $theme = 'basic';
                }

                $options['label_theme'] = $theme;
                break;
        }

        update_option( 'woonutri_options', $options );

        wp_send_json_success( [
            'message' => __( 'Settings saved', 'woonutri' ),
            'next_url' => admin_url( 'admin.php?page=woonutri-setup&step=' . ( $step + 1 ) ),
        ] );
    }

    public static function ajax_complete_wizard() {
        check_ajax_referer( 'woonutri_wizard', 'nonce' );

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error( [ 'message' => __( 'Permission denied', 'woonutri' ) ] );
        }

        $step = isset( $_POST['step'] ) ? absint( $_POST['step'] ) : 4;
        $options = get_option( 'woonutri_options', [] );

        update_option( 'woonutri_wizard_completed', true );
        update_option( 'woonutri_wizard_completed_at', current_time( 'mysql' ) );

        wp_send_json_success( [
            'message' => __( 'Setup complete!', 'woonutri' ),
            'redirect_url' => admin_url( 'admin.php?page=woonutri-dashboard' ),
        ] );
    }

    public static function ajax_skip_wizard() {
        check_ajax_referer( 'woonutri_wizard', 'nonce' );

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error( [ 'message' => __( 'Permission denied', 'woonutri' ) ] );
        }

        update_option( 'woonutri_wizard_completed', true );
        update_option( 'woonutri_wizard_skipped', true );

        wp_send_json_success( [
            'message' => __( 'Setup skipped', 'woonutri' ),
            'redirect_url' => admin_url( 'admin.php?page=woonutri-settings' ),
        ] );
    }
}
