<?php
/**
 * @package WooNutri
 */

if (!defined('ABSPATH')) {
    exit;
}

class Woonutri_Variation_Admin
{
    public static function init()
    {
        add_action('woocommerce_variation_options', [__CLASS__, 'add_variation_options'], 10, 3);
        add_action('woocommerce_product_after_variable_attributes', [__CLASS__, 'add_variation_nutrition_fields'], 10, 3);
        add_action('woocommerce_save_product_variation', [__CLASS__, 'save_variation_nutrition_data'], 10, 2);
        add_filter('woocommerce_available_variation', [__CLASS__, 'add_variation_data'], 10, 3);
    }

    public static function add_variation_options($loop, $variation_data, $variation)
    {
        $variation_id = $variation->ID;
        $inherit = get_post_meta($variation_id, '_woonutri_inherit_parent', true);
        $has_override = ($inherit === 'override');
        ?>
        <label class="tips" data-tip="<?php esc_attr_e('Enable custom nutrition data for this variation', 'smart-nutrition-labels'); ?>">
            <?php esc_html_e('Custom Nutrition:', 'smart-nutrition-labels'); ?>
            <input type="checkbox" class="woonutri-variation-override-toggle checkbox"
                name="woonutri_variation_override[<?php echo esc_attr($loop); ?>]" <?php checked($has_override); ?> />
        </label>
        <?php
    }

    public static function add_variation_nutrition_fields($loop, $variation_data, $variation)
    {
        $variation_id = $variation->ID;
        $inherit = get_post_meta($variation_id, '_woonutri_inherit_parent', true);
        $has_override = ($inherit === 'override');

        $nutrition_data = Woonutri_Meta_Handler::get_variation_nutrition_data($variation_id);

        $variation_fields = self::get_variation_nutrition_fields();
        ?>
        <div class="woonutri-variation-panel" style="<?php echo !$has_override ? 'display:none;' : ''; ?>">
            <div class="form-row form-row-full woonutri-variation-header">
                <h4>
                    <?php esc_html_e('Nutrition Data (Override)', 'smart-nutrition-labels'); ?>
                </h4>
                <p class="description">
                    <?php esc_html_e('Enter nutrition values specific to this variation. Leave blank to inherit from parent product.', 'smart-nutrition-labels'); ?>
                </p>
            </div>

            <input type="hidden" name="woonutri_variation_id[<?php echo esc_attr($loop); ?>]"
                value="<?php echo esc_attr($variation_id); ?>" />

            <div class="woonutri-variation-fields">
                <?php foreach ($variation_fields as $key => $field): ?>
                    <p class="form-row form-row-first">
                        <label for="woonutri_variation_<?php echo esc_attr($key); ?>_<?php echo esc_attr($loop); ?>">
                            <?php echo esc_html($field['label']); ?>
                            <?php if (!empty($field['unit'])): ?>
                                <span class="woonutri-unit">(
                                    <?php echo esc_html($field['unit']); ?>)
                                </span>
                            <?php endif; ?>
                        </label>
                        <input type="<?php echo esc_attr($field['type'] ?? 'number'); ?>" class="short"
                            name="woonutri_variation_data[<?php echo esc_attr($loop); ?>][<?php echo esc_attr($key); ?>]"
                            id="woonutri_variation_<?php echo esc_attr($key); ?>_<?php echo esc_attr($loop); ?>"
                            value="<?php echo esc_attr($nutrition_data[$key] ?? ''); ?>"
                            placeholder="<?php echo esc_attr($field['placeholder'] ?? ''); ?>" step="any" />
                    </p>
                <?php endforeach; ?>
            </div>
        </div>

        <style>
            .woonutri-variation-panel {
                border: 1px solid #dfe0e0;
                background: #f8f9fa;
                padding: 12px;
                margin: 10px 0;
                border-radius: 4px;
            }

            .woonutri-variation-header h4 {
                margin: 0 0 5px 0;
                color: #1d2327;
            }

            .woonutri-variation-fields {
                display: grid;
                grid-template-columns: repeat(2, 1fr);
                gap: 10px;
            }

            .woonutri-variation-fields .form-row {
                margin: 0;
            }

            .woonutri-unit {
                font-weight: normal;
                color: #777;
                font-size: 11px;
            }
        </style>

        <script type="text/javascript">
            jQuery(function ($) {
                $(document).on('change', '.woonutri-variation-override-toggle', function () {
                    var panel = $(this).closest('.woocommerce_variation').find('.woonutri-variation-panel');
                    if ($(this).is(':checked')) {
                        panel.slideDown(200);
                    } else {
                        panel.slideUp(200);
                    }
                });
            });
        </script>
        <?php
    }

    public static function save_variation_nutrition_data($variation_id, $loop)
    {
        $override_enabled = isset($_POST['woonutri_variation_override'][$loop]);

        $nutrition_data = isset($_POST['woonutri_variation_data'][$loop]) ? $_POST['woonutri_variation_data'][$loop] : [];

        $sanitized_data = [];
        foreach ($nutrition_data as $key => $value) {
            if ($value !== '') {
                $sanitized_data[$key] = sanitize_text_field($value);
            }
        }

        if ($override_enabled) {
            Woonutri_Meta_Handler::set_variation_nutrition_data($variation_id, $sanitized_data, 'override');
        } else {
            update_post_meta($variation_id, '_woonutri_inherit_parent', 'inherit');
        }
    }

    public static function add_variation_data($variation_data, $product, $variation)
    {
        $variation_id = $variation->get_id();

        $nutrition = Woonutri_Meta_Handler::get_variation_nutrition_data($variation_id);

        $variation_data['woonutri_has_data'] = !empty($nutrition);
        $variation_data['woonutri_data'] = $nutrition;
        $variation_data['woonutri_has_override'] = Woonutri_Meta_Handler::has_variation_override($variation_id);

        return $variation_data;
    }

    private static function get_variation_nutrition_fields()
    {
        return [
            'energy_kcal' => [
                'label' => __('Energy', 'smart-nutrition-labels'),
                'unit' => 'kcal',
                'type' => 'number',
                'placeholder' => '250'
            ],
            'fat' => [
                'label' => __('Fat', 'smart-nutrition-labels'),
                'unit' => 'g',
                'type' => 'number',
                'placeholder' => '15.5'
            ],
            'saturated_fat' => [
                'label' => __('Saturated Fat', 'smart-nutrition-labels'),
                'unit' => 'g',
                'type' => 'number',
                'placeholder' => '5.0'
            ],
            'carbs' => [
                'label' => __('Carbohydrates', 'smart-nutrition-labels'),
                'unit' => 'g',
                'type' => 'number',
                'placeholder' => '45.0'
            ],
            'sugar' => [
                'label' => __('Sugar', 'smart-nutrition-labels'),
                'unit' => 'g',
                'type' => 'number',
                'placeholder' => '12.5'
            ],
            'fiber' => [
                'label' => __('Fiber', 'smart-nutrition-labels'),
                'unit' => 'g',
                'type' => 'number',
                'placeholder' => '3.2'
            ],
            'protein' => [
                'label' => __('Protein', 'smart-nutrition-labels'),
                'unit' => 'g',
                'type' => 'number',
                'placeholder' => '8.5'
            ],
            'salt' => [
                'label' => __('Salt', 'smart-nutrition-labels'),
                'unit' => 'g',
                'type' => 'number',
                'placeholder' => '1.5'
            ],
            'sodium' => [
                'label' => __('Sodium', 'smart-nutrition-labels'),
                'unit' => 'mg',
                'type' => 'number',
                'placeholder' => '450'
            ],
            'serving_size' => [
                'label' => __('Serving Size', 'smart-nutrition-labels'),
                'unit' => '',
                'type' => 'text',
                'placeholder' => '1 cup (240ml)'
            ],
        ];
    }
}

add_action('init', ['Woonutri_Variation_Admin', 'init']);
