<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * WooNutri Nutrition API
 * Handles external nutrition database lookups and REST API endpoints
 * 
 * @package WooNutri
 * @since 1.0.0
 */

class Woonutri_Nutrition_API {
    
    /**
     * API endpoints
     */
    const OPENFOODFACTS_API = 'https://world.openfoodfacts.org/api/v0/product/';
    const USDA_API = 'https://api.nal.usda.gov/fdc/v1/foods/search';
    
    /**
     * Initialize the API class
     */
    public static function init() {
        add_action( 'rest_api_init', [ __CLASS__, 'register_routes' ] );
        
        // Register AJAX handlers for admin
        if ( is_admin() ) {
            add_action( 'wp_ajax_woonutri_lookup_barcode', array( __CLASS__, 'ajax_lookup_barcode' ) );
        }
    }
    
    /**
     * Register REST API routes
     */
    public static function register_routes() {
        // Product nutrition endpoint
        register_rest_route( 'woonutri/v1', '/products/(?P<id>\d+)/nutrition', array(
            'methods' => 'GET',
            'callback' => array( __CLASS__, 'get_product_nutrition' ),
            'permission_callback' => '__return_true',
            'args' => array(
                'id' => array(
                    'validate_callback' => function($param, $request, $key) {
                        return is_numeric( $param );
                    }
                ),
            ),
        ) );
        
        // Barcode lookup endpoint (Pro only)
        register_rest_route( 'woonutri/v1', '/barcode/(?P<code>[0-9]+)', array(
            'methods' => 'GET',
            'callback' => array( __CLASS__, 'lookup_barcode_rest' ),
            'permission_callback' => function() {
                return current_user_can( 'edit_products' );
            },
            'args' => array(
                'code' => array(
                    'validate_callback' => function($param, $request, $key) {
                        return preg_match( '/^[0-9]{8,14}$/', $param );
                    }
                ),
            ),
        ) );
    }
    
    /**
     * Get product nutrition data via REST API
     */
    public static function get_product_nutrition( $request ) {
        $product_id = $request['id'];
        
        // Check if product exists
        $product = wc_get_product( $product_id );
        if ( ! $product ) {
            return new WP_Error( 'product_not_found', __( 'Product not found', 'woonutri' ), array( 'status' => 404 ) );
        }
        
        // Get nutrition data
        $nutrition_data = Woonutri_Meta_Handler::get_nutrition_data( $product_id );
        
        return rest_ensure_response( array(
            'product_id' => $product_id,
            'product_name' => $product->get_name(),
            'nutrition' => $nutrition_data
        ) );
    }
    
    /**
     * REST API endpoint for barcode lookup
     */
    public static function lookup_barcode_rest( $request ) {
        $barcode = $request['code'];
        $result = self::lookup_barcode( $barcode );
        
        if ( is_wp_error( $result ) ) {
            return $result;
        }
        
        return rest_ensure_response( $result );
    }
    
    /**
     * Lookup nutrition data by barcode
     * 
     * @param string $barcode Product barcode
     * @return array|WP_Error Nutrition data or error
     */
    public static function lookup_barcode( $barcode ) {
        // Check if pro
        if ( ! woonutri_fs()->can_use_premium_code() ) {
            return new WP_Error( 'not_pro', __( 'This feature requires WooNutri Pro', 'woonutri' ), array( 'status' => 403 ) );
        }
        
        // Validate barcode
        $barcode = sanitize_text_field( $barcode );
        if ( ! preg_match( '/^[0-9]{8,14}$/', $barcode ) ) {
            return new WP_Error( 'invalid_barcode', __( 'Invalid barcode format', 'woonutri' ), array( 'status' => 400 ) );
        }
        
        // Try OpenFoodFacts first
        $result = self::lookup_openfoodfacts( $barcode );
        
        if ( ! is_wp_error( $result ) && ! empty( $result ) ) {
            return $result;
        }
        
        // For demo purposes, return simulated data for specific barcodes
        return self::get_demo_barcode_data( $barcode );
    }
    
    /**
     * Lookup nutrition data from OpenFoodFacts
     * 
     * @param string $barcode Product barcode
     * @return array|WP_Error Nutrition data or error
     */
    private static function lookup_openfoodfacts( $barcode ) {
        $url = self::OPENFOODFACTS_API . $barcode . '.json';
        
        $response = wp_remote_get( $url, array(
            'timeout' => 10,
            'headers' => array(
                'User-Agent' => 'WooNutri/1.0 (WordPress Plugin)'
            )
        ) );
        
        if ( is_wp_error( $response ) ) {
            return $response;
        }
        
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );
        
        if ( empty( $data ) || $data['status'] !== 1 ) {
            return new WP_Error( 'not_found', __( 'Product not found', 'woonutri' ) );
        }
        
        return self::parse_openfoodfacts_data( $data['product'] );
    }
    
    /**
     * Parse OpenFoodFacts data to our format
     * 
     * @param array $product OpenFoodFacts product data
     * @return array Nutrition data
     */
    private static function parse_openfoodfacts_data( $product ) {
        $nutrition = array();
        
        // Map OpenFoodFacts fields to our fields
        $field_map = array(
            'energy-kcal_100g' => 'energy_kcal',
            'fat_100g' => 'fat',
            'saturated-fat_100g' => 'saturated_fat',
            'trans-fat_100g' => 'trans_fat',
            'cholesterol_100g' => 'cholesterol',
            'carbohydrates_100g' => 'carbs',
            'sugars_100g' => 'sugar',
            'fiber_100g' => 'fiber',
            'proteins_100g' => 'protein',
            'salt_100g' => 'salt',
            'sodium_100g' => 'sodium',
        );
        
        // Extract nutrition values
        if ( isset( $product['nutriments'] ) ) {
            foreach ( $field_map as $off_field => $our_field ) {
                if ( isset( $product['nutriments'][$off_field] ) ) {
                    $nutrition[$our_field] = $product['nutriments'][$off_field];
                }
            }
        }
        
        // Extract ingredients
        if ( isset( $product['ingredients_text'] ) ) {
            $nutrition['ingredients'] = $product['ingredients_text'];
        }
        
        // Extract allergens
        if ( isset( $product['allergens'] ) ) {
            $nutrition['allergens'] = $product['allergens'];
        }
        
        // Add product name and brand for reference
        if ( isset( $product['product_name'] ) ) {
            $nutrition['product_name'] = $product['product_name'];
        }
        
        if ( isset( $product['brands'] ) ) {
            $nutrition['brand'] = $product['brands'];
        }
        
        return $nutrition;
    }
    
    /**
     * Get demo barcode data for testing
     * 
     * @param string $barcode Product barcode
     * @return array|WP_Error Nutrition data
     */
    private static function get_demo_barcode_data( $barcode ) {
        $demo_products = array(
            '8901030865278' => array(
                'product_name' => 'Organic Granola Bar',
                'brand' => 'Nature Valley',
                'energy_kcal' => '190',
                'fat' => '7',
                'saturated_fat' => '1',
                'trans_fat' => '0',
                'cholesterol' => '0',
                'sodium' => '95',
                'carbs' => '28',
                'fiber' => '3',
                'sugar' => '11',
                'added_sugars' => '6',
                'protein' => '4',
                'vitamins' => 'Vitamin E: 10%, Iron: 4%',
                'ingredients' => 'Whole grain oats, sugar, canola oil, rice flour, honey, salt, brown sugar syrup, baking soda, soy lecithin, natural flavor',
                'allergens' => 'Contains: Soy. May contain: Tree nuts, Milk'
            ),
            '5000159404259' => array(
                'product_name' => 'Chocolate Bar',
                'brand' => 'Mars',
                'energy_kcal' => '230',
                'fat' => '9',
                'saturated_fat' => '4.7',
                'trans_fat' => '0',
                'cholesterol' => '8',
                'sodium' => '42',
                'carbs' => '35',
                'fiber' => '1',
                'sugar' => '30',
                'added_sugars' => '29',
                'protein' => '2.3',
                'ingredients' => 'Sugar, glucose syrup, skimmed milk powder, cocoa butter, cocoa mass, sunflower oil, milk fat, lactose, whey powder, palm fat, emulsifier (soya lecithin), vanilla extract',
                'allergens' => 'Contains: Milk, Soy. May contain: Peanuts'
            )
        );
        
        if ( isset( $demo_products[$barcode] ) ) {
            return $demo_products[$barcode];
        }
        
        return new WP_Error( 'not_found', __( 'Product not found in demo database', 'woonutri' ), array( 'status' => 404 ) );
    }
    
    /**
     * AJAX handler for barcode lookup
     */
    public static function ajax_lookup_barcode() {
        // Verify nonce
        check_ajax_referer( 'woonutri_admin', 'nonce' );
        
        // Check permissions
        if ( ! current_user_can( 'edit_products' ) ) {
            wp_die( -1 );
        }
        
        // Get barcode
        $barcode = isset( $_POST['barcode'] ) ? sanitize_text_field( $_POST['barcode'] ) : '';
        
        if ( empty( $barcode ) ) {
            wp_send_json_error( array( 'message' => __( 'Barcode is required', 'woonutri' ) ) );
        }
        
        // Lookup barcode
        $result = self::lookup_barcode( $barcode );
        
        if ( is_wp_error( $result ) ) {
            wp_send_json_error( array( 'message' => $result->get_error_message() ) );
        }
        
        wp_send_json_success( $result );
    }
} 