<?php
/**
 * @package WooNutri
 * @since 1.1.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Woonutri_Gutenberg_Blocks
{

    
    public static function init()
    {
        add_action('init', [__CLASS__, 'register_blocks']);
        add_action('enqueue_block_editor_assets', [__CLASS__, 'enqueue_editor_assets']);
    }

    
    public static function register_blocks()
    {
        
        if (!function_exists('register_block_type')) {
            return;
        }

        
        register_block_type('woonutri/nutrition-label', [
            'editor_script' => 'woonutri-block-editor',
            'editor_style' => 'woonutri-block-editor',
            'render_callback' => [__CLASS__, 'render_nutrition_label_block'],
            'attributes' => [
                'productId' => [
                    'type' => 'number',
                    'default' => 0,
                ],
                'theme' => [
                    'type' => 'string',
                    'default' => '',
                ],
                'showTitle' => [
                    'type' => 'boolean',
                    'default' => true,
                ],
                'expandable' => [
                    'type' => 'boolean',
                    'default' => true,
                ],
            ],
        ]);

        
        register_block_type('woonutri/allergen-icons', [
            'editor_script' => 'woonutri-block-editor',
            'editor_style' => 'woonutri-block-editor',
            'render_callback' => [__CLASS__, 'render_allergen_icons_block'],
            'attributes' => [
                'productId' => [
                    'type' => 'number',
                    'default' => 0,
                ],
                'showLabel' => [
                    'type' => 'boolean',
                    'default' => true,
                ],
                'size' => [
                    'type' => 'string',
                    'default' => 'medium',
                ],
            ],
        ]);

        
        register_block_type('woonutri/product-comparison', [
            'editor_script' => 'woonutri-block-editor',
            'editor_style' => 'woonutri-block-editor',
            'render_callback' => [__CLASS__, 'render_comparison_block'],
            'attributes' => [
                'productIds' => [
                    'type' => 'string',
                    'default' => '',
                ],
                'fields' => [
                    'type' => 'string',
                    'default' => 'energy_kcal,fat,saturated_fat,carbohydrates,sugar,fiber,protein,salt',
                ],
                'showImage' => [
                    'type' => 'boolean',
                    'default' => true,
                ],
                'showPrice' => [
                    'type' => 'boolean',
                    'default' => true,
                ],
            ],
        ]);
    }

    
    public static function enqueue_editor_assets()
    {
        wp_enqueue_script(
            'woonutri-block-editor',
            WOONUTRI_PLUGIN_URL . 'assets/js/blocks/nutrition-label.js',
            ['wp-blocks', 'wp-element', 'wp-editor', 'wp-components', 'wp-i18n', 'wp-data'],
            WOONUTRI_VERSION,
            true
        );

        wp_enqueue_style(
            'woonutri-block-editor',
            WOONUTRI_PLUGIN_URL . 'assets/css/blocks/nutrition-label-editor.css',
            ['wp-edit-blocks'],
            WOONUTRI_VERSION
        );

        
        wp_localize_script('woonutri-block-editor', 'woonutriBlockData', [
            'pluginUrl' => WOONUTRI_PLUGIN_URL,
            'isPro' => function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code(),
            'upgradeUrl' => function_exists('woonutri_fs') ? woonutri_fs()->get_upgrade_url() : '',
            'themes' => self::get_available_themes(),
            'products' => self::get_products_for_dropdown(),
            'i18n' => [
                'blockTitle' => __('Nutrition Label', 'smart-nutrition-labels'),
                'blockDescription' => __('Display nutrition facts for a product.', 'smart-nutrition-labels'),
                'selectProduct' => __('Select a product', 'smart-nutrition-labels'),
                'noProducts' => __('No products found', 'smart-nutrition-labels'),
                'theme' => __('Theme', 'smart-nutrition-labels'),
                'showTitle' => __('Show Title', 'smart-nutrition-labels'),
                'expandable' => __('Expandable Sections', 'smart-nutrition-labels'),
                'proOnly' => __('Pro Only', 'smart-nutrition-labels'),
                'preview' => __('Preview', 'smart-nutrition-labels'),
                'noProductSelected' => __('Select a product to display its nutrition label.', 'smart-nutrition-labels'),
                'upgradeForThemes' => __('Upgrade to Pro to unlock more themes', 'smart-nutrition-labels'),
            ],
        ]);
    }

    
    private static function get_available_themes()
    {
        $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();

        $themes = [
            [
                'value' => '',
                'label' => __('Default (from settings)', 'smart-nutrition-labels'),
                'pro' => false,
            ],
            [
                'value' => 'basic',
                'label' => __('Basic', 'smart-nutrition-labels'),
                'pro' => false,
            ],
            [
                'value' => 'fda',
                'label' => __('FDA', 'smart-nutrition-labels'),
                'pro' => true,
            ],
            [
                'value' => 'eu',
                'label' => __('EU Compliant', 'smart-nutrition-labels'),
                'pro' => true,
            ],
            [
                'value' => 'uk',
                'label' => __('UK', 'smart-nutrition-labels'),
                'pro' => true,
            ],
            [
                'value' => 'modern',
                'label' => __('Modern', 'smart-nutrition-labels'),
                'pro' => true,
            ],
            [
                'value' => 'dark',
                'label' => __('Dark', 'smart-nutrition-labels'),
                'pro' => true,
            ],
            [
                'value' => 'premium',
                'label' => __('Premium', 'smart-nutrition-labels'),
                'pro' => true,
            ],
            [
                'value' => 'clean',
                'label' => __('Clean', 'smart-nutrition-labels'),
                'pro' => true,
            ],
            [
                'value' => 'minimal',
                'label' => __('Minimal', 'smart-nutrition-labels'),
                'pro' => true,
            ],
        ];

        
        if (!$is_pro) {
            foreach ($themes as &$theme) {
                if ($theme['pro']) {
                    $theme['disabled'] = true;
                    $theme['label'] .= ' (' . __('Pro', 'smart-nutrition-labels') . ')';
                }
            }
        }

        return $themes;
    }

    
    private static function get_products_for_dropdown()
    {
        $products = get_posts([
            'post_type' => 'product',
            'post_status' => 'publish',
            'posts_per_page' => 100,
            'orderby' => 'title',
            'order' => 'ASC',
        ]);

        $options = [
            [
                'value' => 0,
                'label' => __('— Select Product —', 'smart-nutrition-labels'),
            ],
        ];

        foreach ($products as $product) {
            $options[] = [
                'value' => $product->ID,
                'label' => $product->post_title,
            ];
        }

        return $options;
    }

    
    public static function render_nutrition_label_block($attributes)
    {
        $product_id = isset($attributes['productId']) ? absint($attributes['productId']) : 0;
        $theme = isset($attributes['theme']) ? sanitize_text_field($attributes['theme']) : '';
        $show_title = isset($attributes['showTitle']) ? (bool) $attributes['showTitle'] : true;
        $expandable = isset($attributes['expandable']) ? (bool) $attributes['expandable'] : true;

        
        if (!$product_id && is_singular('product')) {
            $product_id = get_the_ID();
        }

        if (!$product_id) {
            return '<p class="woonutri-block-placeholder">' .
                esc_html__('No product selected. Edit this block to select a product.', 'smart-nutrition-labels') .
                '</p>';
        }

        
        $data = \Woonutri_Meta_Handler::get_nutrition_data($product_id);
        if (empty(array_filter($data))) {
            return '<p class="woonutri-block-placeholder">' .
                esc_html__('This product has no nutrition data.', 'smart-nutrition-labels') .
                '</p>';
        }

        
        wp_enqueue_style('woonutri-label', WOONUTRI_PLUGIN_URL . 'assets/css/label.css', [], WOONUTRI_VERSION);
        wp_enqueue_style('woonutri-label-fda', WOONUTRI_PLUGIN_URL . 'assets/css/label-fda.css', ['woonutri-label'], WOONUTRI_VERSION);
        wp_enqueue_style('woonutri-label-eu', WOONUTRI_PLUGIN_URL . 'assets/css/label-eu.css', ['woonutri-label'], WOONUTRI_VERSION);
        wp_enqueue_script('woonutri-nutrition-label', WOONUTRI_PLUGIN_URL . 'assets/js/nutrition-label.js', ['jquery'], WOONUTRI_VERSION, true);

        
        return \Woonutri_Label_Renderer::render_label($product_id, [], [
            'theme' => $theme,
            'show_title' => $show_title,
            'expandable' => $expandable,
        ]);
    }

    
    public static function render_allergen_icons_block($attributes)
    {
        $product_id = isset($attributes['productId']) ? absint($attributes['productId']) : 0;
        $show_label = isset($attributes['showLabel']) ? (bool) $attributes['showLabel'] : true;
        $size = isset($attributes['size']) ? sanitize_text_field($attributes['size']) : 'medium';

        
        if (!$product_id && is_singular('product')) {
            $product_id = get_the_ID();
        }

        if (!$product_id) {
            return '<p class="woonutri-block-placeholder">' .
                esc_html__('No product selected. Edit this block to select a product.', 'smart-nutrition-labels') .
                '</p>';
        }

        
        $data = \Woonutri_Meta_Handler::get_nutrition_data($product_id);
        $allergens = isset($data['allergens']) ? $data['allergens'] : [];

        if (empty($allergens)) {
            return '<p class="woonutri-block-placeholder">' .
                esc_html__('This product has no allergen data.', 'smart-nutrition-labels') .
                '</p>';
        }

        
        $allergen_text = is_array($allergens) ? implode(', ', $allergens) : $allergens;

        
        wp_enqueue_style('woonutri-comparison', WOONUTRI_PLUGIN_URL . 'assets/css/comparison.css', [], WOONUTRI_VERSION);

        
        $size_class = 'woonutri-allergen-size-' . $size;

        
        $output = '<div class="woonutri-allergen-block ' . esc_attr($size_class) . '">';

        if ($show_label) {
            $output .= '<span class="woonutri-allergen-label">' . esc_html__('Contains:', 'smart-nutrition-labels') . ' </span>';
        }

        
        if (class_exists('Woonutri_Traffic_Lights')) {
            $output .= \Woonutri_Traffic_Lights::render_allergen_icons($allergen_text);
        } else {
            
            $output .= '<span class="woonutri-allergen-list">' . esc_html($allergen_text) . '</span>';
        }

        $output .= '</div>';

        return $output;
    }

    
    public static function render_comparison_block($attributes)
    {
        
        if (!function_exists('woonutri_fs') || !woonutri_fs()->can_use_premium_code()) {
            return '<div class="woonutri-block-pro-notice">' .
                '<p>' . esc_html__('Product Comparison is a Pro feature.', 'smart-nutrition-labels') . '</p>' .
                '<a href="' . esc_url(woonutri_fs()->get_upgrade_url()) . '" class="button">' .
                esc_html__('Upgrade to Pro', 'smart-nutrition-labels') . '</a>' .
                '</div>';
        }

        $product_ids_str = isset($attributes['productIds']) ? sanitize_text_field($attributes['productIds']) : '';
        $fields_str = isset($attributes['fields']) ? sanitize_text_field($attributes['fields']) : 'energy_kcal,fat,saturated_fat,carbohydrates,sugar,fiber,protein,salt';
        $show_image = isset($attributes['showImage']) ? (bool) $attributes['showImage'] : true;
        $show_price = isset($attributes['showPrice']) ? (bool) $attributes['showPrice'] : true;

        if (empty($product_ids_str)) {
            return '<p class="woonutri-block-placeholder">' .
                esc_html__('No products selected. Edit this block to add products for comparison.', 'smart-nutrition-labels') .
                '</p>';
        }

        $product_ids = array_filter(array_map('intval', explode(',', $product_ids_str)));
        $fields = array_filter(array_map('trim', explode(',', $fields_str)));

        if (count($product_ids) < 2) {
            return '<p class="woonutri-block-placeholder">' .
                esc_html__('Please select at least 2 products to compare.', 'smart-nutrition-labels') .
                '</p>';
        }

        
        wp_enqueue_style('woonutri-comparison', WOONUTRI_PLUGIN_URL . 'assets/css/comparison.css', [], WOONUTRI_VERSION);

        
        if (class_exists('Woonutri_Product_Comparison')) {
            return \Woonutri_Product_Comparison::render_comparison_table($product_ids, $fields, $show_image, $show_price);
        }

        return '<p class="woonutri-block-placeholder">' .
            esc_html__('Comparison functionality not available.', 'smart-nutrition-labels') .
            '</p>';
    }
}
