<?php
class Woonutri_Label_Standards {

    public static function get_daily_values( $standard = 'fda' ) {
        $daily_values = array(
            'fda' => array(
                'energy_kcal' => 2000,
                'fat' => 78,
                'saturated_fat' => 20,
                'trans_fat' => 0,
                'cholesterol' => 300,
                'sodium' => 2300,
                'carbs' => 275,
                'fiber' => 28,
                'sugar' => 50,
                'added_sugars' => 50,
                'protein' => 50,
                'potassium' => 2600,
                'calcium' => 1300,
                'iron' => 18,
                'vitamin_d' => 20,
                'vitamin_a' => 900,
                'vitamin_c' => 90,
                'vitamin_e' => 15,
                'vitamin_k' => 120,
                'thiamin' => 1.2,
                'riboflavin' => 1.3,
                'niacin' => 16,
                'vitamin_b6' => 1.7,
                'folate' => 400,
                'vitamin_b12' => 2.4,
                'biotin' => 30,
                'pantothenic_acid' => 5,
                'phosphorus' => 1250,
                'iodine' => 150,
                'magnesium' => 420,
                'zinc' => 11,
                'selenium' => 55,
                'copper' => 0.9,
                'manganese' => 2.3,
                'chromium' => 35,
                'molybdenum' => 45,
                'chloride' => 2300
            ),
            'eu' => array(
                'energy_kcal' => 2000,
                'energy_kj' => 8400,
                'fat' => 70,
                'saturated_fat' => 20,
                'carbs' => 260,
                'sugar' => 90,
                'protein' => 50,
                'salt' => 6,
                'fiber' => 25,
                'vitamin_a' => 800,
                'vitamin_d' => 5,
                'vitamin_e' => 12,
                'vitamin_k' => 75,
                'vitamin_c' => 80,
                'thiamin' => 1.1,
                'riboflavin' => 1.4,
                'niacin' => 16,
                'vitamin_b6' => 1.4,
                'folic_acid' => 200,
                'vitamin_b12' => 2.5,
                'biotin' => 50,
                'pantothenic_acid' => 6,
                'potassium' => 2000,
                'chloride' => 800,
                'calcium' => 800,
                'phosphorus' => 700,
                'magnesium' => 375,
                'iron' => 14,
                'zinc' => 10,
                'copper' => 1,
                'manganese' => 2,
                'fluoride' => 3.5,
                'selenium' => 55,
                'chromium' => 40,
                'molybdenum' => 50,
                'iodine' => 150
            ),
            'uk' => array(
                'energy_kcal' => 2000,
                'energy_kj' => 8400,
                'fat' => 70,
                'saturates' => 20,
                'carbohydrate' => 260,
                'sugars' => 90,
                'protein' => 50,
                'salt' => 6,
                'fibre' => 30,
                'vitamin_a' => 800,
                'vitamin_d' => 5,
                'vitamin_e' => 12,
                'vitamin_k' => 75,
                'vitamin_c' => 80,
                'thiamine' => 1.1,
                'riboflavin' => 1.4,
                'niacin' => 16,
                'vitamin_b6' => 1.4,
                'folic_acid' => 200,
                'vitamin_b12' => 2.5,
                'biotin' => 50,
                'pantothenic_acid' => 6,
                'potassium' => 2000,
                'chloride' => 800,
                'calcium' => 800,
                'phosphorus' => 700,
                'magnesium' => 375,
                'iron' => 14,
                'zinc' => 10,
                'copper' => 1,
                'manganese' => 2,
                'fluoride' => 3.5,
                'selenium' => 55,
                'chromium' => 40,
                'molybdenum' => 50,
                'iodine' => 150
            )
        );

        return isset( $daily_values[$standard] ) ? $daily_values[$standard] : $daily_values['fda'];
    }

    public static function calculate_daily_value_percentage( $value, $nutrient, $standard = 'fda' ) {
        $daily_values = self::get_daily_values( $standard );

        if ( !isset( $daily_values[$nutrient] ) || $daily_values[$nutrient] == 0 ) {
            return null;
        }

        $percentage = round( ( $value / $daily_values[$nutrient] ) * 100 );

        return $percentage . '%';
    }

    public static function get_required_nutrients( $standard = 'fda' ) {
        $required = array(
            'fda' => array(
                'energy_kcal',
                'fat',
                'saturated_fat',
                'trans_fat',
                'cholesterol',
                'sodium',
                'carbs',
                'fiber',
                'sugar',
                'added_sugars',
                'protein'
            ),
            'eu' => array(
                'energy_kcal',
                'energy_kj',
                'fat',
                'saturated_fat',
                'carbs',
                'sugar',
                'protein',
                'salt'
            ),
            'uk' => array(
                'energy_kcal',
                'energy_kj',
                'fat',
                'saturates',
                'carbohydrate',
                'sugars',
                'protein',
                'salt'
            )
        );

        return isset( $required[$standard] ) ? $required[$standard] : $required['fda'];
    }

    public static function get_field_mapping( $standard = 'fda' ) {
        $mapping = array(
            'fda' => array(
                'energy_kcal' => array( 'label' => 'Calories', 'unit' => '', 'bold' => true ),
                'fat' => array( 'label' => 'Total Fat', 'unit' => 'g', 'show_dv' => true ),
                'saturated_fat' => array( 'label' => 'Saturated Fat', 'unit' => 'g', 'show_dv' => true, 'indent' => true ),
                'trans_fat' => array( 'label' => 'Trans Fat', 'unit' => 'g', 'indent' => true ),
                'cholesterol' => array( 'label' => 'Cholesterol', 'unit' => 'mg', 'show_dv' => true ),
                'sodium' => array( 'label' => 'Sodium', 'unit' => 'mg', 'show_dv' => true ),
                'carbs' => array( 'label' => 'Total Carbohydrate', 'unit' => 'g', 'show_dv' => true ),
                'fiber' => array( 'label' => 'Dietary Fiber', 'unit' => 'g', 'show_dv' => true, 'indent' => true ),
                'sugar' => array( 'label' => 'Total Sugars', 'unit' => 'g', 'indent' => true ),
                'added_sugars' => array( 'label' => 'Includes Added Sugars', 'unit' => 'g', 'show_dv' => true, 'indent' => 2 ),
                'protein' => array( 'label' => 'Protein', 'unit' => 'g', 'show_dv' => true ),
                'vitamin_d' => array( 'label' => 'Vitamin D', 'unit' => 'mcg', 'show_dv' => true ),
                'calcium' => array( 'label' => 'Calcium', 'unit' => 'mg', 'show_dv' => true ),
                'iron' => array( 'label' => 'Iron', 'unit' => 'mg', 'show_dv' => true ),
                'potassium' => array( 'label' => 'Potassium', 'unit' => 'mg', 'show_dv' => true )
            ),
            'eu' => array(
                'energy_kj' => array( 'label' => 'Energy', 'unit' => 'kJ', 'show_ri' => true ),
                'energy_kcal' => array( 'label' => '', 'unit' => 'kcal', 'show_ri' => true, 'same_line' => true ),
                'fat' => array( 'label' => 'Fat', 'unit' => 'g', 'show_ri' => true ),
                'saturated_fat' => array( 'label' => 'of which saturates', 'unit' => 'g', 'show_ri' => true, 'indent' => true ),
                'carbs' => array( 'label' => 'Carbohydrate', 'unit' => 'g', 'show_ri' => true ),
                'sugar' => array( 'label' => 'of which sugars', 'unit' => 'g', 'show_ri' => true, 'indent' => true ),
                'fiber' => array( 'label' => 'Fibre', 'unit' => 'g' ),
                'protein' => array( 'label' => 'Protein', 'unit' => 'g', 'show_ri' => true ),
                'salt' => array( 'label' => 'Salt', 'unit' => 'g', 'show_ri' => true )
            ),
            'uk' => array(
                'energy_kj' => array( 'label' => 'Energy', 'unit' => 'kJ', 'show_ri' => true ),
                'energy_kcal' => array( 'label' => '', 'unit' => 'kcal', 'show_ri' => true, 'same_line' => true ),
                'fat' => array( 'label' => 'Fat', 'unit' => 'g', 'show_ri' => true ),
                'saturates' => array( 'label' => 'Saturates', 'unit' => 'g', 'show_ri' => true ),
                'carbohydrate' => array( 'label' => 'Carbohydrate', 'unit' => 'g', 'show_ri' => true ),
                'sugars' => array( 'label' => 'Sugars', 'unit' => 'g', 'show_ri' => true ),
                'fibre' => array( 'label' => 'Fibre', 'unit' => 'g' ),
                'protein' => array( 'label' => 'Protein', 'unit' => 'g', 'show_ri' => true ),
                'salt' => array( 'label' => 'Salt', 'unit' => 'g', 'show_ri' => true )
            )
        );

        return isset( $mapping[$standard] ) ? $mapping[$standard] : $mapping['fda'];
    }

    public static function convert_to_standard( $data, $to_standard = 'fda' ) {
        $converted = $data;

        if ( isset( $data['energy_kcal'] ) && !isset( $data['energy_kj'] ) ) {
            $converted['energy_kj'] = round( $data['energy_kcal'] * 4.184 );
        }

        if ( isset( $data['sodium'] ) && !isset( $data['salt'] ) ) {
            $converted['salt'] = round( $data['sodium'] / 400, 1 );
        }

        if ( isset( $data['salt'] ) && !isset( $data['sodium'] ) ) {
            $converted['sodium'] = round( $data['salt'] * 400 );
        }

        if ( $to_standard === 'uk' ) {
            if ( isset( $data['saturated_fat'] ) ) {
                $converted['saturates'] = $data['saturated_fat'];
            }
            if ( isset( $data['carbs'] ) ) {
                $converted['carbohydrate'] = $data['carbs'];
            }
            if ( isset( $data['sugar'] ) ) {
                $converted['sugars'] = $data['sugar'];
            }
            if ( isset( $data['fiber'] ) ) {
                $converted['fibre'] = $data['fiber'];
            }
        }

        return $converted;
    }

    public static function get_serving_text( $standard = 'fda', $product_data = array() ) {
        $serving_size = isset( $product_data['serving_size'] ) ? $product_data['serving_size'] : '100g';
        $servings_per_container = isset( $product_data['servings_per_container'] ) ? $product_data['servings_per_container'] : '';

        switch ( $standard ) {
            case 'fda':
                $text = sprintf( __( 'Serving Size %s', 'woonutri' ), $serving_size );
                if ( $servings_per_container ) {
                    $text .= "\n" . sprintf( __( 'Servings Per Container %s', 'woonutri' ), $servings_per_container );
                }
                return $text;

            case 'eu':
            case 'uk':
                return sprintf( __( 'Typical values per %s', 'woonutri' ), $serving_size );

            default:
                return sprintf( __( 'Per %s', 'woonutri' ), $serving_size );
        }
    }

    public static function format_value( $value, $nutrient, $standard = 'fda' ) {
        switch ( $standard ) {
            case 'fda':
                if ( $nutrient === 'energy_kcal' ) {
                    if ( $value < 5 ) return '0';
                    if ( $value <= 50 ) return round( $value / 5 ) * 5;
                    return round( $value / 10 ) * 10;
                }
                if ( in_array( $nutrient, array( 'fat', 'saturated_fat', 'trans_fat' ) ) ) {
                    if ( $value < 0.5 ) return '0';
                    if ( $value < 5 ) return round( $value, 1 );
                    return round( $value );
                }
                if ( in_array( $nutrient, array( 'cholesterol', 'sodium', 'potassium' ) ) ) {
                    if ( $value < 5 ) return '0';
                    if ( $value <= 140 ) return round( $value / 5 ) * 5;
                    return round( $value / 10 ) * 10;
                }
                if ( in_array( $nutrient, array( 'carbs', 'fiber', 'sugar', 'protein' ) ) ) {
                    if ( $value < 0.5 ) return '0';
                    return round( $value );
                }
                break;

            case 'eu':
            case 'uk':
                if ( $nutrient === 'energy_kj' || $nutrient === 'energy_kcal' ) {
                    return round( $value );
                }
                if ( in_array( $nutrient, array( 'fat', 'saturated_fat', 'saturates', 'carbs', 'carbohydrate', 'sugar', 'sugars', 'fiber', 'fibre', 'protein' ) ) ) {
                    if ( $value < 0.5 ) return '0';
                    if ( $value < 10 ) return round( $value, 1 );
                    return round( $value );
                }
                if ( $nutrient === 'salt' ) {
                    if ( $value < 0.01 ) return '0';
                    if ( $value < 1 ) return round( $value, 2 );
                    return round( $value, 1 );
                }
                break;
        }

        return $value;
    }

    public static function get_compliance_warnings( $data, $standard = 'fda' ) {
        $warnings = array();
        $required = self::get_required_nutrients( $standard );

        foreach ( $required as $nutrient ) {
            if ( !isset( $data[$nutrient] ) || $data[$nutrient] === '' ) {
                $field_map = self::get_field_mapping( $standard );
                $label = isset( $field_map[$nutrient]['label'] ) ? $field_map[$nutrient]['label'] : $nutrient;
                $warnings[] = sprintf( __( 'Missing required field: %s', 'woonutri' ), $label );
            }
        }

        switch ( $standard ) {
            case 'fda':
                if ( !isset( $data['serving_size'] ) || empty( $data['serving_size'] ) ) {
                    $warnings[] = __( 'FDA requires serving size to be specified', 'woonutri' );
                }
                break;

            case 'eu':
            case 'uk':
                if ( isset( $data['energy_kcal'] ) && !isset( $data['energy_kj'] ) ) {
                    $warnings[] = __( 'EU/UK requires energy to be displayed in both kJ and kcal', 'woonutri' );
                }
                break;
        }

        return $warnings;
    }
}
