<?php
class Woonutri_Nutrient_Combinations {

    private static $predefined_combinations = [
        'basic' => [
            'name' => 'Basic Nutrition Facts',
            'description' => 'Essential nutrition information',
            'nutrients' => [
                'energy_kcal', 'fat', 'saturated_fat', 'carbs', 'sugar', 'protein', 'salt'
            ],
            'display_nested' => true
        ],
        'fda_required' => [
            'name' => 'FDA Required Nutrients',
            'description' => 'All FDA mandated nutrition facts',
            'nutrients' => [
                'energy_kcal', 'fat', 'saturated_fat', 'trans_fat', 'cholesterol', 'sodium',
                'carbs', 'fiber', 'sugar', 'added_sugars', 'protein', 'vitamin_d', 'calcium', 'iron', 'potassium'
            ],
            'display_nested' => true
        ],
        'eu_standard' => [
            'name' => 'EU Standard Nutrition',
            'description' => 'EU regulation compliant nutrition facts',
            'nutrients' => [
                'energy_kcal', 'energy_kj', 'fat', 'saturated_fat', 'carbs', 'sugar', 'fiber', 'protein', 'salt'
            ],
            'display_nested' => true
        ],
        'vitamins_minerals' => [
            'name' => 'Vitamins & Minerals',
            'description' => 'Comprehensive vitamin and mineral profile',
            'nutrients' => [
                'vitamin_a', 'vitamin_c', 'vitamin_d', 'vitamin_e', 'vitamin_k',
                'thiamin', 'riboflavin', 'niacin', 'vitamin_b6', 'folate', 'vitamin_b12',
                'calcium', 'iron', 'magnesium', 'phosphorus', 'potassium', 'sodium', 'zinc'
            ],
            'display_nested' => false
        ],
        'sports_nutrition' => [
            'name' => 'Sports Nutrition',
            'description' => 'Nutrition facts for athletic performance',
            'nutrients' => [
                'energy_kcal', 'protein', 'carbs', 'sugar', 'fiber', 'fat', 'sodium', 'potassium',
                'caffeine', 'creatine', 'bcaa_total'
            ],
            'display_nested' => false
        ],
        'weight_management' => [
            'name' => 'Weight Management',
            'description' => 'Key nutrients for weight management',
            'nutrients' => [
                'energy_kcal', 'fat', 'saturated_fat', 'carbs', 'fiber', 'sugar', 'protein'
            ],
            'display_nested' => true
        ],
        'heart_healthy' => [
            'name' => 'Heart Healthy Focus',
            'description' => 'Nutrients important for cardiovascular health',
            'nutrients' => [
                'energy_kcal', 'fat', 'saturated_fat', 'trans_fat', 'cholesterol', 'sodium',
                'fiber', 'omega_3', 'potassium', 'magnesium'
            ],
            'display_nested' => true
        ],
        'diabetic_friendly' => [
            'name' => 'Diabetic Friendly',
            'description' => 'Focus on carbohydrates and sugar content',
            'nutrients' => [
                'energy_kcal', 'carbs', 'fiber', 'sugar', 'added_sugars', 'protein', 'fat', 'sodium'
            ],
            'display_nested' => true
        ],
        'supplement_facts' => [
            'name' => 'Supplement Facts',
            'description' => 'Comprehensive nutrient profile for supplements',
            'nutrients' => [
                'serving_size', 'servings_per_container',
                'vitamin_a', 'vitamin_c', 'vitamin_d', 'vitamin_e', 'vitamin_k',
                'thiamin', 'riboflavin', 'niacin', 'vitamin_b6', 'folate', 'vitamin_b12', 'biotin', 'pantothenic_acid',
                'calcium', 'iron', 'phosphorus', 'iodine', 'magnesium', 'zinc', 'selenium', 'copper', 'manganese', 'chromium', 'molybdenum'
            ],
            'display_nested' => false
        ]
    ];

    public static function get_available_combinations() {
        $combinations = self::$predefined_combinations;

        $custom_combinations = get_option( 'woonutri_custom_combinations', [] );
        if ( !empty( $custom_combinations ) ) {
            $combinations = array_merge( $combinations, $custom_combinations );
        }

        return apply_filters( 'woonutri_nutrient_combinations', $combinations );
    }

    public static function get_combination( $combination_key ) {
        $combinations = self::get_available_combinations();
        return isset( $combinations[ $combination_key ] ) ? $combinations[ $combination_key ] : null;
    }

    public static function create_custom_combination( $key, $name, $description, $nutrients, $display_nested = true ) {
        $custom_combinations = get_option( 'woonutri_custom_combinations', [] );

        $custom_combinations[ $key ] = [
            'name' => sanitize_text_field( $name ),
            'description' => sanitize_textarea_field( $description ),
            'nutrients' => array_map( 'sanitize_key', $nutrients ),
            'display_nested' => (bool) $display_nested,
            'custom' => true,
            'created_date' => current_time( 'mysql' )
        ];

        update_option( 'woonutri_custom_combinations', $custom_combinations );

        return $custom_combinations[ $key ];
    }

    public static function delete_custom_combination( $key ) {
        $custom_combinations = get_option( 'woonutri_custom_combinations', [] );

        if ( isset( $custom_combinations[ $key ] ) ) {
            unset( $custom_combinations[ $key ] );
            update_option( 'woonutri_custom_combinations', $custom_combinations );
            return true;
        }

        return false;
    }

    public static function get_display_nutrients( $combination_key, $product_data ) {
        $combination = self::get_combination( $combination_key );

        if ( !$combination ) {
            return [];
        }

        $hierarchy = \Woonutri_Meta_Handler::get_field_hierarchy();
        $display_nutrients = [];

        foreach ( $combination['nutrients'] as $nutrient_key ) {
            if ( isset( $product_data[ $nutrient_key ] ) && $product_data[ $nutrient_key ] !== '' ) {
                $nutrient_config = $hierarchy[ $nutrient_key ] ?? [];

                $display_nutrients[ $nutrient_key ] = [
                    'key' => $nutrient_key,
                    'label' => $nutrient_config['label'] ?? ucfirst( str_replace( '_', ' ', $nutrient_key ) ),
                    'value' => $product_data[ $nutrient_key ],
                    'unit' => $nutrient_config['unit'] ?? 'g',
                    'type' => $nutrient_config['type'] ?? 'nutrient',
                    'category' => $nutrient_config['category'] ?? 'basic',
                    'children' => $nutrient_config['children'] ?? []
                ];

                if ( $combination['display_nested'] && !empty( $nutrient_config['children'] ) ) {
                    $display_nutrients[ $nutrient_key ]['children'] = self::get_child_nutrients(
                        $nutrient_config['children'],
                        $product_data,
                        $hierarchy
                    );
                }
            }
        }

        return $display_nutrients;
    }

    private static function get_child_nutrients( $children_config, $product_data, $hierarchy ) {
        $children = [];

        foreach ( $children_config as $child_key => $child_config ) {
            if ( isset( $product_data[ $child_key ] ) && $product_data[ $child_key ] !== '' ) {
                $children[ $child_key ] = [
                    'key' => $child_key,
                    'label' => $child_config['label'] ?? ucfirst( str_replace( '_', ' ', $child_key ) ),
                    'value' => $product_data[ $child_key ],
                    'unit' => $child_config['unit'] ?? 'g',
                    'type' => $child_config['type'] ?? 'nested',
                    'category' => $child_config['category'] ?? 'basic'
                ];

                if ( !empty( $child_config['children'] ) ) {
                    $children[ $child_key ]['children'] = self::get_child_nutrients(
                        $child_config['children'],
                        $product_data,
                        $hierarchy
                    );
                }
            }
        }

        return $children;
    }

    public static function get_nutrients_by_category() {
        $hierarchy = \Woonutri_Meta_Handler::get_field_hierarchy();
        $categories = [];

        foreach ( $hierarchy as $key => $config ) {
            $category = $config['category'] ?? 'uncategorized';
            $categories[ $category ][ $key ] = $config;
        }

        $category_order = [
            'basic', 'required_fda', 'macronutrient', 'fat_soluble', 'water_soluble',
            'b_complex', 'major', 'trace', 'amino_acids', 'specialty', 'carotenoid',
            'bioactive', 'eu_system', 'au_system', 'uk_system', 'product_info', 'supplementary'
        ];

        $ordered_categories = [];
        foreach ( $category_order as $cat ) {
            if ( isset( $categories[ $cat ] ) ) {
                $ordered_categories[ $cat ] = $categories[ $cat ];
            }
        }

        foreach ( $categories as $cat => $nutrients ) {
            if ( !isset( $ordered_categories[ $cat ] ) ) {
                $ordered_categories[ $cat ] = $nutrients;
            }
        }

        return $ordered_categories;
    }

    public static function get_category_display_names() {
        return [
            'basic' => __( 'Basic Nutrition', 'woonutri' ),
            'required_fda' => __( 'FDA Required', 'woonutri' ),
            'macronutrient' => __( 'Macronutrients', 'woonutri' ),
            'fat_soluble' => __( 'Fat-Soluble Vitamins', 'woonutri' ),
            'water_soluble' => __( 'Water-Soluble Vitamins', 'woonutri' ),
            'b_complex' => __( 'B-Complex Vitamins', 'woonutri' ),
            'major' => __( 'Major Minerals', 'woonutri' ),
            'trace' => __( 'Trace Minerals', 'woonutri' ),
            'amino_acids' => __( 'Amino Acids', 'woonutri' ),
            'specialty' => __( 'Specialty Nutrients', 'woonutri' ),
            'carotenoid' => __( 'Carotenoids', 'woonutri' ),
            'bioactive' => __( 'Bioactive Compounds', 'woonutri' ),
            'eu_system' => __( 'EU Rating System', 'woonutri' ),
            'au_system' => __( 'Australian System', 'woonutri' ),
            'uk_system' => __( 'UK System', 'woonutri' ),
            'product_info' => __( 'Product Information', 'woonutri' ),
            'supplementary' => __( 'Supplementary Info', 'woonutri' )
        ];
    }

    public static function render_combination_selector( $current_combination = '', $field_name = 'nutrient_combination' ) {
        $combinations = self::get_available_combinations();
        ?>
        <select name="<?php echo esc_attr( $field_name ); ?>" id="<?php echo esc_attr( $field_name ); ?>" class="woonutri-combination-selector">
            <option value=""><?php _e( 'Select a nutrient combination...', 'woonutri' ); ?></option>

            <?php foreach ( $combinations as $key => $combo ): ?>
                <option value="<?php echo esc_attr( $key ); ?>"
                        <?php selected( $current_combination, $key ); ?>
                        data-description="<?php echo esc_attr( $combo['description'] ); ?>"
                        data-nutrient-count="<?php echo count( $combo['nutrients'] ); ?>"
                        data-nested="<?php echo $combo['display_nested'] ? 'true' : 'false'; ?>">
                    <?php echo esc_html( $combo['name'] ); ?>
                    (<?php echo count( $combo['nutrients'] ); ?> <?php _e( 'nutrients', 'woonutri' ); ?>)
                </option>
            <?php endforeach; ?>

            <option value="custom"><?php _e( 'Create Custom Combination...', 'woonutri' ); ?></option>
        </select>

        <div id="combination-description" class="description" style="margin-top: 5px; font-style: italic; color: #666;"></div>

        <script>
        document.addEventListener('DOMContentLoaded', function() {
            const selector = document.getElementById('<?php echo esc_js( $field_name ); ?>');
            const description = document.getElementById('combination-description');

            selector.addEventListener('change', function() {
                const option = this.options[this.selectedIndex];
                if (option.dataset.description) {
                    description.textContent = option.dataset.description;
                } else {
                    description.textContent = '';
                }
            });

            selector.dispatchEvent(new Event('change'));
        });
        </script>
        <?php
    }

    public static function render_custom_combination_builder() {
        $categories = self::get_nutrients_by_category();
        $category_names = self::get_category_display_names();
        ?>
        <div id="custom-combination-builder" style="display: none; margin-top: 20px; padding: 20px; border: 1px solid #ddd; background: #f9f9f9;">
            <h3><?php _e( 'Create Custom Nutrient Combination', 'woonutri' ); ?></h3>

            <table class="form-table">
                <tr>
                    <th><label for="custom_combo_name"><?php _e( 'Combination Name', 'woonutri' ); ?></label></th>
                    <td><input type="text" id="custom_combo_name" name="custom_combo_name" class="regular-text" /></td>
                </tr>
                <tr>
                    <th><label for="custom_combo_description"><?php _e( 'Description', 'woonutri' ); ?></label></th>
                    <td><textarea id="custom_combo_description" name="custom_combo_description" rows="3" class="large-text"></textarea></td>
                </tr>
                <tr>
                    <th><label><?php _e( 'Display Options', 'woonutri' ); ?></label></th>
                    <td>
                        <label>
                            <input type="checkbox" id="custom_combo_nested" name="custom_combo_nested" value="1" checked />
                            <?php _e( 'Enable nested display for supported nutrients', 'woonutri' ); ?>
                        </label>
                    </td>
                </tr>
            </table>

            <h4><?php _e( 'Select Nutrients by Category', 'woonutri' ); ?></h4>
            <div class="nutrient-categories">
                <?php foreach ( $categories as $category_key => $nutrients ): ?>
                    <div class="nutrient-category" style="margin-bottom: 20px; padding: 15px; border: 1px solid #e0e0e0; background: white;">
                        <h5 style="margin-top: 0;">
                            <label>
                                <input type="checkbox" class="category-toggle" data-category="<?php echo esc_attr( $category_key ); ?>" />
                                <?php echo esc_html( $category_names[ $category_key ] ?? ucfirst( str_replace( '_', ' ', $category_key ) ) ); ?>
                                <span style="color: #666; font-weight: normal;">(<?php echo count( $nutrients ); ?> <?php _e( 'nutrients', 'woonutri' ); ?>)</span>
                            </label>
                        </h5>

                        <div class="nutrient-list" style="margin-left: 20px; display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 10px;">
                            <?php foreach ( $nutrients as $nutrient_key => $nutrient_config ): ?>
                                <label style="display: flex; align-items: center; font-size: 13px;">
                                    <input type="checkbox"
                                           name="custom_combo_nutrients[]"
                                           value="<?php echo esc_attr( $nutrient_key ); ?>"
                                           data-category="<?php echo esc_attr( $category_key ); ?>"
                                           style="margin-right: 5px;" />
                                    <?php echo esc_html( $nutrient_config['label'] ?? ucfirst( str_replace( '_', ' ', $nutrient_key ) ) ); ?>
                                </label>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>

            <p class="submit">
                <button type="button" class="button-primary" id="save-custom-combination">
                    <?php _e( 'Save Custom Combination', 'woonutri' ); ?>
                </button>
                <button type="button" class="button-secondary" id="cancel-custom-combination">
                    <?php _e( 'Cancel', 'woonutri' ); ?>
                </button>
            </p>
        </div>

        <script>
        document.addEventListener('DOMContentLoaded', function() {
            const selector = document.querySelector('.woonutri-combination-selector');
            const builder = document.getElementById('custom-combination-builder');

            if (selector) {
                selector.addEventListener('change', function() {
                    if (this.value === 'custom') {
                        builder.style.display = 'block';
                    } else {
                        builder.style.display = 'none';
                    }
                });
            }

            document.querySelectorAll('.category-toggle').forEach(function(toggle) {
                toggle.addEventListener('change', function() {
                    const category = this.dataset.category;
                    const checkboxes = document.querySelectorAll(`input[data-category="${category}"]`);
                    checkboxes.forEach(function(checkbox) {
                        if (checkbox !== toggle) {
                            checkbox.checked = toggle.checked;
                        }
                    });
                });
            });

            document.getElementById('cancel-custom-combination')?.addEventListener('click', function() {
                builder.style.display = 'none';
                if (selector) {
                    selector.value = '';
                }
            });
        });
        </script>
        <?php
    }

    public static function get_combination_stats() {
        $combinations = self::get_available_combinations();
        $stats = [
            'total_combinations' => count( $combinations ),
            'predefined' => count( self::$predefined_combinations ),
            'custom' => count( get_option( 'woonutri_custom_combinations', [] ) ),
            'most_popular' => null,
            'avg_nutrients_per_combo' => 0
        ];

        $total_nutrients = 0;
        foreach ( $combinations as $combo ) {
            $total_nutrients += count( $combo['nutrients'] );
        }
        $stats['avg_nutrients_per_combo'] = $stats['total_combinations'] > 0
            ? round( $total_nutrients / $stats['total_combinations'], 1 )
            : 0;

        return $stats;
    }
}
