<?php
/**
 * @package WooNutri
 * @since 2.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Woonutri_PDF_Export
{
    
    private static $dompdf = null;

    
    const LABEL_SIZES = [
        'small' => [
            'name' => 'Small (50x70mm)',
            'width' => 50,
            'height' => 70,
            'font_scale' => 0.7,
            'description' => 'Suitable for small packages, jars, bottles'
        ],
        'medium' => [
            'name' => 'Medium (70x100mm)',
            'width' => 70,
            'height' => 100,
            'font_scale' => 0.85,
            'description' => 'Standard food packaging label'
        ],
        'large' => [
            'name' => 'Large (100x150mm)',
            'width' => 100,
            'height' => 150,
            'font_scale' => 1.0,
            'description' => 'Large packages, boxes, retail display'
        ],
        'fda_standard' => [
            'name' => 'FDA Standard (2.625" x 5")',
            'width' => 66.7,  
            'height' => 127,  
            'font_scale' => 0.9,
            'description' => 'US FDA standard nutrition facts panel'
        ],
        'fda_tabular' => [
            'name' => 'FDA Tabular (5" x 2.5")',
            'width' => 127,
            'height' => 63.5,
            'font_scale' => 0.85,
            'description' => 'Horizontal/tabular format for wide packages'
        ],
        'eu_standard' => [
            'name' => 'EU Standard (60x90mm)',
            'width' => 60,
            'height' => 90,
            'font_scale' => 0.8,
            'description' => 'European Union nutrition declaration format'
        ],
        'full_page' => [
            'name' => 'Full Page (A4/Letter)',
            'width' => 210,
            'height' => 297,
            'font_scale' => 1.0,
            'description' => 'Full page for documentation or large displays'
        ],
        'custom' => [
            'name' => 'Custom Size',
            'width' => 70,
            'height' => 100,
            'font_scale' => 1.0,
            'description' => 'Define your own dimensions'
        ]
    ];

    
    const LABEL_TEMPLATES = [
        
        'standard' => [
            'name' => 'Standard',
            'description' => 'Clean, professional nutrition label',
            'type' => 'pdf',
            'border' => true,
            'rounded' => false
        ],
        'fda_2020' => [
            'name' => 'FDA 2020 Format',
            'description' => 'Updated FDA nutrition facts format with bold calories',
            'type' => 'pdf',
            'border' => true,
            'rounded' => false
        ],
        'eu_table' => [
            'name' => 'EU Table Format',
            'description' => 'European tabular nutrition declaration',
            'type' => 'pdf',
            'border' => true,
            'rounded' => false
        ],
        'traffic_light' => [
            'name' => 'UK Traffic Light',
            'description' => 'UK-style with color-coded indicators',
            'type' => 'pdf',
            'border' => true,
            'rounded' => true
        ],
    ];

    
    public static function get_all_themes()
    {
        
        $themes = self::LABEL_TEMPLATES;

        
        $visual_themes = [
            
            'basic' => ['name' => 'WooNutri', 'description' => 'Default plugin theme', 'type' => 'visual', 'group' => 'free'],
            'minimal' => ['name' => 'Minimal', 'description' => 'Clean minimal design', 'type' => 'visual', 'group' => 'free'],
            'clean' => ['name' => 'Clean', 'description' => 'Simple clean look', 'type' => 'visual', 'group' => 'free'],
            'simple' => ['name' => 'Simple', 'description' => 'Straightforward layout', 'type' => 'visual', 'group' => 'free'],
            'compact' => ['name' => 'Compact', 'description' => 'Space-efficient design', 'type' => 'visual', 'group' => 'free'],

            
            'fda' => ['name' => 'FDA', 'description' => 'US FDA format', 'type' => 'visual', 'group' => 'regional'],
            'eu' => ['name' => 'EU', 'description' => 'European Union format', 'type' => 'visual', 'group' => 'regional'],
            'uk' => ['name' => 'UK', 'description' => 'UK format', 'type' => 'visual', 'group' => 'regional'],
            'australia' => ['name' => 'Australia', 'description' => 'Australian format', 'type' => 'visual', 'group' => 'regional'],
            'canada' => ['name' => 'Canada', 'description' => 'Canadian format', 'type' => 'visual', 'group' => 'regional'],
            'nutriscore' => ['name' => 'Nutri-Score', 'description' => 'French Nutri-Score format', 'type' => 'visual', 'group' => 'regional'],
            'latin' => ['name' => 'Latin', 'description' => 'Latin American format', 'type' => 'visual', 'group' => 'regional'],
            'gda' => ['name' => 'GDA', 'description' => 'Guideline Daily Amount', 'type' => 'visual', 'group' => 'regional'],

            
            'modern' => ['name' => 'Modern', 'description' => 'Contemporary design', 'type' => 'visual', 'group' => 'design'],
            'classic' => ['name' => 'Classic', 'description' => 'Traditional look', 'type' => 'visual', 'group' => 'design'],
            'elegant' => ['name' => 'Elegant', 'description' => 'Refined styling', 'type' => 'visual', 'group' => 'design'],
            'bold' => ['name' => 'Bold', 'description' => 'Strong visual impact', 'type' => 'visual', 'group' => 'design'],
            'soft' => ['name' => 'Soft', 'description' => 'Gentle appearance', 'type' => 'visual', 'group' => 'design'],
            'sharp' => ['name' => 'Sharp', 'description' => 'Crisp edges', 'type' => 'visual', 'group' => 'design'],
            'rounded' => ['name' => 'Rounded', 'description' => 'Curved corners', 'type' => 'visual', 'group' => 'design'],
            'flat' => ['name' => 'Flat', 'description' => 'Flat design style', 'type' => 'visual', 'group' => 'design'],

            
            'organic' => ['name' => 'Organic', 'description' => 'Natural organic style', 'type' => 'visual', 'group' => 'specialty'],
            'fitness' => ['name' => 'Fitness', 'description' => 'Sports nutrition style', 'type' => 'visual', 'group' => 'specialty'],
            'kids' => ['name' => 'Kids', 'description' => 'Child-friendly design', 'type' => 'visual', 'group' => 'specialty'],
            'premium' => ['name' => 'Premium', 'description' => 'Luxury look', 'type' => 'visual', 'group' => 'specialty'],
            'eco' => ['name' => 'Eco', 'description' => 'Eco-friendly style', 'type' => 'visual', 'group' => 'specialty'],
            'medical' => ['name' => 'Medical', 'description' => 'Clinical style', 'type' => 'visual', 'group' => 'specialty'],
            'scientific' => ['name' => 'Scientific', 'description' => 'Lab-style format', 'type' => 'visual', 'group' => 'specialty'],
            'vegan' => ['name' => 'Vegan', 'description' => 'Plant-based style', 'type' => 'visual', 'group' => 'specialty'],

            
            'gradient' => ['name' => 'Gradient', 'description' => 'Colorful gradient style', 'type' => 'visual', 'group' => 'creative'],
            'neon' => ['name' => 'Neon', 'description' => 'Glowing neon effect', 'type' => 'visual', 'group' => 'creative'],
            'retro' => ['name' => 'Retro', 'description' => 'Vintage look', 'type' => 'visual', 'group' => 'creative'],
            'dark' => ['name' => 'Dark', 'description' => 'Dark mode style', 'type' => 'visual', 'group' => 'creative'],
            'paper' => ['name' => 'Paper', 'description' => 'Paper document style', 'type' => 'visual', 'group' => 'creative'],
            'glass' => ['name' => 'Glass', 'description' => 'Frosted glass effect', 'type' => 'visual', 'group' => 'creative'],
            'chalk' => ['name' => 'Chalk', 'description' => 'Chalkboard style', 'type' => 'visual', 'group' => 'creative'],
            'blueprint' => ['name' => 'Blueprint', 'description' => 'Technical blueprint', 'type' => 'visual', 'group' => 'creative'],
        ];

        return array_merge($themes, $visual_themes);
    }

    
    public static function init()
    {
        
        add_action('wp_ajax_woonutri_export_pdf', [__CLASS__, 'ajax_export_pdf']);
        add_action('wp_ajax_nopriv_woonutri_export_pdf', [__CLASS__, 'ajax_export_pdf_public']);
        add_action('wp_ajax_woonutri_export_pdf_batch', [__CLASS__, 'ajax_export_pdf_batch']);
        add_action('wp_ajax_woonutri_preview_pdf', [__CLASS__, 'ajax_preview_pdf']);

        
        add_action('woonutri_after_label', [__CLASS__, 'render_export_button'], 10, 2);
        add_action('woonutri_admin_after_nutrition_tab', [__CLASS__, 'render_admin_export_panel']);

        
        add_action('admin_menu', [__CLASS__, 'add_export_page'], 20);

        
        add_action('admin_enqueue_scripts', [__CLASS__, 'enqueue_admin_scripts']);

        
        add_action('woonutri_cleanup_pdf', [__CLASS__, 'cleanup_pdf']);

        
        add_filter('post_row_actions', [__CLASS__, 'add_product_row_action'], 10, 2);
        add_filter('page_row_actions', [__CLASS__, 'add_product_row_action'], 10, 2);

        
        add_filter('bulk_actions-edit-product', [__CLASS__, 'add_bulk_export_action']);
        add_filter('handle_bulk_actions-edit-product', [__CLASS__, 'handle_bulk_export_action'], 10, 3);

        
        add_action('admin_notices', [__CLASS__, 'bulk_export_admin_notice']);

        
        add_filter('manage_edit-product_columns', [__CLASS__, 'add_nutrition_column'], 20);
        add_action('manage_product_posts_custom_column', [__CLASS__, 'render_nutrition_column'], 10, 2);
        add_filter('manage_edit-product_sortable_columns', [__CLASS__, 'make_nutrition_column_sortable']);
        add_action('pre_get_posts', [__CLASS__, 'sort_by_nutrition_column']);

        
        add_action('admin_head', [__CLASS__, 'add_column_styles']);

        
        add_action('admin_post_woonutri_quick_export_pdf', [__CLASS__, 'handle_quick_export']);
    }

    
    public static function add_export_page()
    {
        add_submenu_page(
            'woonutri-dashboard',
            __('Label Export', 'smart-nutrition-labels'),
            __('Label Export', 'smart-nutrition-labels'),
            'manage_woocommerce',
            'woonutri-export',
            [__CLASS__, 'render_export_page']
        );
    }

    
    public static function enqueue_admin_scripts($hook)
    {
        if (strpos($hook, 'woonutri-export') === false && strpos($hook, 'post.php') === false) {
            return;
        }

        wp_enqueue_style(
            'woonutri-pdf-export',
            WOONUTRI_PLUGIN_URL . 'assets/css/pdf-export.css',
            [],
            WOONUTRI_VERSION
        );

        wp_enqueue_script(
            'woonutri-pdf-export',
            WOONUTRI_PLUGIN_URL . 'assets/js/pdf-export.js',
            ['jquery'],
            WOONUTRI_VERSION,
            true
        );

        wp_localize_script('woonutri-pdf-export', 'woonutriPdfExport', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('woonutri_pdf_export'),
            'label_sizes' => self::LABEL_SIZES,
            'label_templates' => self::LABEL_TEMPLATES,
            'i18n' => [
                'generating' => __('Generating PDF...', 'smart-nutrition-labels'),
                'download' => __('Download PDF', 'smart-nutrition-labels'),
                'preview' => __('Preview', 'smart-nutrition-labels'),
                'error' => __('Error generating PDF', 'smart-nutrition-labels'),
                'select_products' => __('Please select at least one product', 'smart-nutrition-labels'),
                'batch_progress' => __('Generating %d of %d...', 'smart-nutrition-labels'),
                'batch_complete' => __('Batch export complete!', 'smart-nutrition-labels'),
            ]
        ]);
    }

    
    public static function is_dompdf_available()
    {
        if (class_exists('Dompdf\Dompdf')) {
            return true;
        }

        $autoload = WOONUTRI_PLUGIN_DIR . 'vendor/autoload.php';
        if (file_exists($autoload)) {
            require_once $autoload;
            return class_exists('Dompdf\Dompdf');
        }

        return false;
    }

    
    private static function get_dompdf($options = [])
    {
        if (!self::is_dompdf_available()) {
            return null;
        }

        $dompdf = new \Dompdf\Dompdf([
            'isRemoteEnabled' => true,
            'isHtml5ParserEnabled' => true,
            'defaultFont' => 'helvetica',
            'dpi' => isset($options['dpi']) ? $options['dpi'] : 300,
            'defaultMediaType' => 'print',
            'isFontSubsettingEnabled' => true,
        ]);

        return $dompdf;
    }

    
    public static function generate_pdf($html, $options = [])
    {
        $dompdf = self::get_dompdf($options);

        if (!$dompdf) {
            return false;
        }

        $defaults = [
            'size' => 'medium',
            'orientation' => 'portrait',
            'dpi' => 300,
            'include_bleed' => false,
            'bleed_mm' => 3,
            'include_crop_marks' => false,
        ];
        $options = wp_parse_args($options, $defaults);

        
        $label_size = isset(self::LABEL_SIZES[$options['size']])
            ? self::LABEL_SIZES[$options['size']]
            : self::LABEL_SIZES['medium'];

        
        $width = $label_size['width'];
        $height = $label_size['height'];

        if ($options['include_bleed']) {
            $width += ($options['bleed_mm'] * 2);
            $height += ($options['bleed_mm'] * 2);
        }

        
        $width_pt = $width * 2.834645669;
        $height_pt = $height * 2.834645669;

        
        $full_html = self::wrap_html($html, $options, $label_size);

        try {
            $dompdf->loadHtml($full_html);
            $dompdf->setPaper([0, 0, $width_pt, $height_pt], $options['orientation']);
            $dompdf->render();

            
            if ($options['include_crop_marks']) {
                self::add_crop_marks($dompdf, $options);
            }

            return $dompdf->output();
        } catch (\Exception) {
            return false;
        }
    }

    
    private static function wrap_html($content, $options, $label_size)
    {
        $css = self::get_pdf_styles($options, $label_size);
        $bleed_padding = $options['include_bleed'] ? $options['bleed_mm'] . 'mm' : '0';

        return '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>' . esc_html__('Nutrition Facts Label', 'smart-nutrition-labels') . '</title>
    <style>' . $css . '</style>
</head>
<body>
    <div class="pdf-page" style="padding: ' . $bleed_padding . ';">
        <div class="pdf-container">
            ' . $content . '
        </div>
    </div>
</body>
</html>';
    }

    
    private static function get_pdf_styles($options, $label_size)
    {
        $font_scale = $label_size['font_scale'];
        $template = isset($options['template']) ? $options['template'] : 'standard';
        $template_config = isset(self::LABEL_TEMPLATES[$template])
            ? self::LABEL_TEMPLATES[$template]
            : self::LABEL_TEMPLATES['standard'];

        
        $title_size = round(24 * $font_scale);
        $subtitle_size = round(14 * $font_scale);
        $body_size = round(10 * $font_scale);
        $small_size = round(8 * $font_scale);
        $calories_size = round(36 * $font_scale);

        $border_style = $template_config['border'] ? '2px solid #000' : 'none';
        $border_radius = $template_config['rounded'] ? '4px' : '0';

        return '
        /* Reset & Base */
        @page {
            margin: 0;
            padding: 0;
        }

        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }

        body {
            font-family: "Helvetica Neue", Helvetica, Arial, sans-serif;
            font-size: ' . $body_size . 'pt;
            line-height: 1.3;
            color: #000;
            background: #fff;
            -webkit-print-color-adjust: exact;
            print-color-adjust: exact;
        }

        .pdf-page {
            width: 100%;
            height: 100%;
            background: #fff;
        }

        .pdf-container {
            width: 100%;
            height: 100%;
        }

        /* ===== NUTRITION LABEL STYLES ===== */

        .woonutri-label,
        .woonutri-pdf-label {
            border: ' . $border_style . ';
            border-radius: ' . $border_radius . ';
            padding: 8px;
            background: #fff;
            width: 100%;
            max-width: 100%;
        }

        /* Title Section */
        .woonutri-label-title,
        .nutrition-title {
            font-size: ' . $title_size . 'pt;
            font-weight: 900;
            margin: 0 0 2px 0;
            text-align: left;
            letter-spacing: -0.5px;
            font-family: "Franklin Gothic Heavy", "Arial Black", sans-serif;
        }

        .woonutri-label-subtitle {
            font-size: ' . $subtitle_size . 'pt;
            font-weight: bold;
            margin-bottom: 4px;
        }

        /* Serving Info */
        .woonutri-serving-info,
        .serving-info {
            font-size: ' . $body_size . 'pt;
            border-bottom: 8px solid #000;
            padding-bottom: 4px;
            margin-bottom: 4px;
        }

        .serving-size-line {
            display: flex;
            justify-content: space-between;
            font-weight: bold;
        }

        .servings-per-container {
            font-weight: normal;
        }

        /* Calories Section (FDA 2020 style) */
        .calories-section {
            display: flex;
            justify-content: space-between;
            align-items: baseline;
            border-bottom: 4px solid #000;
            padding: 4px 0;
            margin-bottom: 4px;
        }

        .calories-label {
            font-size: ' . $subtitle_size . 'pt;
            font-weight: bold;
        }

        .calories-value {
            font-size: ' . $calories_size . 'pt;
            font-weight: 900;
            font-family: "Franklin Gothic Heavy", "Arial Black", sans-serif;
        }

        /* Daily Value Header */
        .dv-header {
            text-align: right;
            font-size: ' . $small_size . 'pt;
            font-weight: bold;
            border-bottom: 1px solid #000;
            padding: 2px 0;
        }

        /* Nutrient Rows */
        .woonutri-label-row,
        .nutrient-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 2px 0;
            border-bottom: 1px solid #ddd;
            font-size: ' . $body_size . 'pt;
        }

        .nutrient-row.major,
        .woonutri-label-row.major {
            border-bottom: 1px solid #000;
        }

        .nutrient-row.thick-bottom {
            border-bottom: 8px solid #000;
        }

        .nutrient-row.medium-bottom {
            border-bottom: 4px solid #000;
        }

        .nutrient-name {
            flex: 1;
            font-weight: normal;
        }

        .nutrient-name.bold,
        .nutrient-row.bold .nutrient-name {
            font-weight: bold;
        }

        .nutrient-name.indent {
            padding-left: 12px;
        }

        .nutrient-name.indent-2 {
            padding-left: 24px;
        }

        .nutrient-value {
            text-align: right;
            min-width: 35px;
            font-weight: normal;
        }

        .nutrient-dv {
            text-align: right;
            min-width: 30px;
            font-weight: bold;
        }

        /* Footer Notes */
        .label-footer,
        .dv-footnote {
            font-size: ' . $small_size . 'pt;
            margin-top: 6px;
            padding-top: 4px;
            border-top: 1px solid #000;
            line-height: 1.2;
        }

        .dv-footnote p {
            margin: 0 0 2px 0;
        }

        /* ===== TRAFFIC LIGHT STYLES (UK) ===== */

        .traffic-lights-container {
            display: flex;
            justify-content: space-around;
            margin: 8px 0;
            padding: 6px;
            background: #f5f5f5;
            border-radius: 4px;
        }

        .traffic-light-item {
            text-align: center;
            padding: 6px 8px;
            border-radius: 4px;
            min-width: 50px;
        }

        .traffic-light-item.green {
            background: #d4edda;
            border: 2px solid #28a745;
        }

        .traffic-light-item.amber {
            background: #fff3cd;
            border: 2px solid #ffc107;
        }

        .traffic-light-item.red {
            background: #f8d7da;
            border: 2px solid #dc3545;
        }

        .traffic-light-indicator {
            width: 14px;
            height: 14px;
            border-radius: 50%;
            display: inline-block;
            margin-bottom: 2px;
        }

        .traffic-light-item.green .traffic-light-indicator { background: #28a745; }
        .traffic-light-item.amber .traffic-light-indicator { background: #ffc107; }
        .traffic-light-item.red .traffic-light-indicator { background: #dc3545; }

        .traffic-light-label {
            font-size: ' . $small_size . 'pt;
            font-weight: bold;
            display: block;
        }

        .traffic-light-value {
            font-size: ' . $body_size . 'pt;
            font-weight: bold;
        }

        /* ===== NUTRI-SCORE ===== */

        .nutri-score-container {
            margin: 8px 0;
            text-align: center;
        }

        .nutri-score-bar {
            display: inline-flex;
            border-radius: 4px;
            overflow: hidden;
        }

        .nutri-score-letter {
            width: 28px;
            height: 28px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
            font-size: ' . $subtitle_size . 'pt;
            color: #fff;
        }

        .nutri-score-letter.score-a { background: #038141; }
        .nutri-score-letter.score-b { background: #85BB2F; }
        .nutri-score-letter.score-c { background: #FECB02; color: #000; }
        .nutri-score-letter.score-d { background: #EE8100; }
        .nutri-score-letter.score-e { background: #E63946; }

        .nutri-score-letter.active {
            transform: scale(1.3);
            border-radius: 4px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.3);
        }

        /* ===== ALLERGEN SECTION ===== */

        .allergen-section {
            margin: 6px 0;
            padding: 6px;
            background: #fff3cd;
            border: 1px solid #ffc107;
            border-radius: 4px;
            font-size: ' . $small_size . 'pt;
        }

        .allergen-title {
            font-weight: bold;
            margin-bottom: 2px;
        }

        .allergen-list {
            font-weight: bold;
        }

        /* ===== EU TABLE FORMAT ===== */

        .eu-nutrition-table {
            width: 100%;
            border-collapse: collapse;
            font-size: ' . $body_size . 'pt;
        }

        .eu-nutrition-table th,
        .eu-nutrition-table td {
            border: 1px solid #000;
            padding: 3px 5px;
            text-align: left;
        }

        .eu-nutrition-table th {
            background: #f0f0f0;
            font-weight: bold;
        }

        .eu-nutrition-table td.value {
            text-align: right;
        }

        /* ===== INGREDIENTS SECTION ===== */

        .ingredients-section {
            margin-top: 6px;
            padding-top: 4px;
            border-top: 1px solid #000;
            font-size: ' . $small_size . 'pt;
        }

        .ingredients-title {
            font-weight: bold;
        }

        .ingredients-list {
            text-align: justify;
        }

        .allergen-bold {
            font-weight: bold;
            text-transform: uppercase;
        }

        /* ===== PRINT MARKS ===== */

        .crop-marks {
            position: absolute;
            stroke: #000;
            stroke-width: 0.25pt;
        }

        .bleed-area {
            position: absolute;
            border: 0.5pt dashed #ccc;
        }

        /* ===== PRODUCT INFO HEADER ===== */

        .product-header {
            margin-bottom: 8px;
            padding-bottom: 4px;
            border-bottom: 1px solid #ccc;
        }

        .product-name {
            font-size: ' . $subtitle_size . 'pt;
            font-weight: bold;
            margin: 0;
        }

        .product-sku {
            font-size: ' . $small_size . 'pt;
            color: #666;
        }

        /* ===== BARCODE PLACEHOLDER ===== */

        .barcode-area {
            margin-top: 8px;
            text-align: center;
        }

        .barcode-placeholder {
            display: inline-block;
            padding: 4px 8px;
            border: 1px dashed #999;
            font-size: ' . $small_size . 'pt;
            color: #999;
        }

        /* ===== REGULATORY MARKS ===== */

        .regulatory-marks {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-top: 6px;
            padding-top: 4px;
            border-top: 1px solid #ddd;
            font-size: ' . $small_size . 'pt;
        }

        .best-before {
            font-weight: bold;
        }

        .batch-code {
            color: #666;
        }
        ';
    }

    
    private static function add_crop_marks($dompdf, $options)
    {
        
        
        $canvas = $dompdf->getCanvas();
        $bleed = $options['bleed_mm'] * 2.834645669;
        $mark_length = 10;
        $mark_offset = 5;

        
        $w = $canvas->get_width();
        $h = $canvas->get_height();

        
        $canvas->set_stroke_color([0, 0, 0]);
        $canvas->set_line_width(0.5);

        
        $canvas->line($bleed - $mark_offset - $mark_length, $bleed, $bleed - $mark_offset, $bleed);
        $canvas->line($bleed, $bleed - $mark_offset - $mark_length, $bleed, $bleed - $mark_offset);

        
        $canvas->line($w - $bleed + $mark_offset, $bleed, $w - $bleed + $mark_offset + $mark_length, $bleed);
        $canvas->line($w - $bleed, $bleed - $mark_offset - $mark_length, $w - $bleed, $bleed - $mark_offset);

        
        $canvas->line($bleed - $mark_offset - $mark_length, $h - $bleed, $bleed - $mark_offset, $h - $bleed);
        $canvas->line($bleed, $h - $bleed + $mark_offset, $bleed, $h - $bleed + $mark_offset + $mark_length);

        
        $canvas->line($w - $bleed + $mark_offset, $h - $bleed, $w - $bleed + $mark_offset + $mark_length, $h - $bleed);
        $canvas->line($w - $bleed, $h - $bleed + $mark_offset, $w - $bleed, $h - $bleed + $mark_offset + $mark_length);
    }

    
    public static function generate_label_html($product_id, $options = [])
    {
        $product = wc_get_product($product_id);
        if (!$product) {
            return '';
        }

        $nutrition_data = Woonutri_Meta_Handler::get_nutrition_data($product_id);
        if (empty($nutrition_data)) {
            return '';
        }

        $defaults = [
            'template' => 'standard',
            'region' => 'us',
            'include_product_header' => false,
            'include_allergens' => true,
            'include_ingredients' => true,
            'include_nutri_score' => false,
            'include_traffic_lights' => false,
            'include_barcode_area' => false,
            'include_regulatory' => false,
        ];
        $options = wp_parse_args($options, $defaults);

        
        $plugin_options = get_option('woonutri_options', []);
        $region = isset($options['region']) ? $options['region'] : (isset($plugin_options['region']) ? $plugin_options['region'] : 'us');

        
        $all_themes = self::get_all_themes();
        $template = $options['template'];
        $is_visual_theme = isset($all_themes[$template]) && isset($all_themes[$template]['type']) && $all_themes[$template]['type'] === 'visual';

        
        if ($is_visual_theme && class_exists('Woonutri_Label_Renderer')) {
            
            $css = self::get_label_renderer_css();

            
            $label_html = Woonutri_Label_Renderer::render_label($product_id, [], [
                'theme' => $template,
                'show_title' => true,
                'expandable' => false, 
            ]);

            if (!empty($label_html)) {
                ob_start();

                
                if ($options['include_product_header']) {
                    echo '<div class="product-header">';
                    echo '<h3 class="product-name">' . esc_html($product->get_name()) . '</h3>';
                    if ($product->get_sku()) {
                        echo '<div class="product-sku">SKU: ' . esc_html($product->get_sku()) . '</div>';
                    }
                    echo '</div>';
                }

                
                echo '<style>' . $css . '</style>';
                echo $label_html;

                
                if ($options['include_barcode_area']) {
                    echo '<div class="barcode-area">';
                    echo '<div class="barcode-placeholder">' . esc_html__('Barcode Area', 'smart-nutrition-labels') . '</div>';
                    echo '</div>';
                }

                
                if ($options['include_regulatory']) {
                    echo self::render_regulatory_info($product, $options);
                }

                return ob_get_clean();
            }
        }

        ob_start();

        
        if ($options['include_product_header']) {
            echo '<div class="product-header">';
            echo '<h3 class="product-name">' . esc_html($product->get_name()) . '</h3>';
            if ($product->get_sku()) {
                echo '<div class="product-sku">SKU: ' . esc_html($product->get_sku()) . '</div>';
            }
            echo '</div>';
        }

        
        switch ($options['template']) {
            case 'fda_2020':
                echo self::render_fda_2020_label($nutrition_data, $options);
                break;
            case 'eu_table':
                echo self::render_eu_table_label($nutrition_data, $options);
                break;
            case 'traffic_light':
                echo self::render_traffic_light_label($nutrition_data, $options);
                break;
            default:
                echo self::render_standard_label($nutrition_data, $options);
        }

        
        if ($options['include_nutri_score'] && isset($nutrition_data['nutri_score'])) {
            echo self::render_nutri_score($nutrition_data['nutri_score']);
        }

        
        if ($options['include_allergens'] && !empty($nutrition_data['allergens'])) {
            echo self::render_allergens($nutrition_data['allergens']);
        }

        
        if ($options['include_ingredients'] && !empty($nutrition_data['ingredients'])) {
            echo self::render_ingredients($nutrition_data['ingredients'], $nutrition_data['allergens'] ?? []);
        }

        
        if ($options['include_barcode_area']) {
            echo '<div class="barcode-area">';
            echo '<div class="barcode-placeholder">' . esc_html__('Barcode Area', 'smart-nutrition-labels') . '</div>';
            echo '</div>';
        }

        
        if ($options['include_regulatory']) {
            echo self::render_regulatory_info($product, $options);
        }

        return ob_get_clean();
    }

    
    private static function render_fda_2020_label($data, $options)
    {
        $serving_size = isset($data['serving_size']) ? $data['serving_size'] : '1 serving';
        $servings = isset($data['servings_per_container']) ? $data['servings_per_container'] : '';
        $calories = isset($data['energy_kcal']) ? $data['energy_kcal'] : '0';

        ob_start();
        ?>
        <div class="woonutri-pdf-label fda-2020">
            <div class="nutrition-title">Nutrition Facts</div>

            <div class="serving-info">
                <?php if ($servings): ?>
                <div class="servings-per-container"><?php echo esc_html($servings); ?> servings per container</div>
                <?php endif; ?>
                <div class="serving-size-line">
                    <span>Serving size</span>
                    <span><?php echo esc_html($serving_size); ?></span>
                </div>
            </div>

            <div class="calories-section">
                <span class="calories-label">Calories</span>
                <span class="calories-value"><?php echo esc_html($calories); ?></span>
            </div>

            <div class="dv-header">% Daily Value*</div>

            <?php
            
            $nutrients = [
                ['key' => 'fat', 'name' => 'Total Fat', 'unit' => 'g', 'dv' => 78, 'bold' => true],
                ['key' => 'saturated_fat', 'name' => 'Saturated Fat', 'unit' => 'g', 'dv' => 20, 'indent' => true],
                ['key' => 'trans_fat', 'name' => 'Trans Fat', 'unit' => 'g', 'dv' => null, 'indent' => true],
                ['key' => 'cholesterol', 'name' => 'Cholesterol', 'unit' => 'mg', 'dv' => 300, 'bold' => true],
                ['key' => 'sodium', 'name' => 'Sodium', 'unit' => 'mg', 'dv' => 2300, 'bold' => true],
                ['key' => 'carbs', 'name' => 'Total Carbohydrate', 'unit' => 'g', 'dv' => 275, 'bold' => true],
                ['key' => 'fiber', 'name' => 'Dietary Fiber', 'unit' => 'g', 'dv' => 28, 'indent' => true],
                ['key' => 'sugar', 'name' => 'Total Sugars', 'unit' => 'g', 'dv' => null, 'indent' => true],
                ['key' => 'added_sugars', 'name' => 'Includes Added Sugars', 'unit' => 'g', 'dv' => 50, 'indent' => true, 'indent_level' => 2],
                ['key' => 'protein', 'name' => 'Protein', 'unit' => 'g', 'dv' => 50, 'bold' => true, 'thick_bottom' => true],
            ];

            foreach ($nutrients as $nutrient) {
                $value = isset($data[$nutrient['key']]) ? $data[$nutrient['key']] : '';
                if ($value === '') continue;

                $dv_percent = '';
                if ($nutrient['dv'] && $value > 0) {
                    $dv_percent = round(($value / $nutrient['dv']) * 100) . '%';
                }

                $classes = ['nutrient-row'];
                if (!empty($nutrient['bold'])) $classes[] = 'bold';
                if (!empty($nutrient['thick_bottom'])) $classes[] = 'thick-bottom';
                if (!empty($nutrient['medium_bottom'])) $classes[] = 'medium-bottom';
                ?>
                <div class="<?php echo esc_attr(implode(' ', $classes)); ?>">
                    <span class="nutrient-name <?php echo !empty($nutrient['indent']) ? 'indent' : ''; ?> <?php echo !empty($nutrient['indent_level']) && $nutrient['indent_level'] == 2 ? 'indent-2' : ''; ?>">
                        <?php echo esc_html($nutrient['name']); ?>
                    </span>
                    <span class="nutrient-value"><?php echo esc_html($value . $nutrient['unit']); ?></span>
                    <span class="nutrient-dv"><?php echo esc_html($dv_percent); ?></span>
                </div>
                <?php
            }

            
            $vitamins = [
                ['key' => 'vitamin_d', 'name' => 'Vitamin D', 'unit' => 'mcg', 'dv' => 20],
                ['key' => 'calcium', 'name' => 'Calcium', 'unit' => 'mg', 'dv' => 1300],
                ['key' => 'iron', 'name' => 'Iron', 'unit' => 'mg', 'dv' => 18],
                ['key' => 'potassium', 'name' => 'Potassium', 'unit' => 'mg', 'dv' => 4700],
            ];

            foreach ($vitamins as $vitamin) {
                $value = isset($data[$vitamin['key']]) ? $data[$vitamin['key']] : '';
                if ($value === '') continue;

                $dv_percent = $vitamin['dv'] > 0 ? round(($value / $vitamin['dv']) * 100) . '%' : '';
                ?>
                <div class="nutrient-row">
                    <span class="nutrient-name"><?php echo esc_html($vitamin['name']); ?></span>
                    <span class="nutrient-value"><?php echo esc_html($value . $vitamin['unit']); ?></span>
                    <span class="nutrient-dv"><?php echo esc_html($dv_percent); ?></span>
                </div>
                <?php
            }
            ?>

            <div class="dv-footnote">
                <p>* The % Daily Value (DV) tells you how much a nutrient in a serving of food contributes to a daily diet. 2,000 calories a day is used for general nutrition advice.</p>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    
    private static function render_eu_table_label($data, $options)
    {
        $serving_size = isset($data['serving_size']) ? $data['serving_size'] : '';

        ob_start();
        ?>
        <div class="woonutri-pdf-label eu-table">
            <div class="nutrition-title">Nutrition Information</div>

            <table class="eu-nutrition-table">
                <thead>
                    <tr>
                        <th><?php esc_html_e('Typical Values', 'smart-nutrition-labels'); ?></th>
                        <th><?php esc_html_e('Per 100g', 'smart-nutrition-labels'); ?></th>
                        <?php if ($serving_size): ?>
                        <th><?php printf(esc_html__('Per %s', 'smart-nutrition-labels'), esc_html($serving_size)); ?></th>
                        <?php endif; ?>
                        <th>%RI*</th>
                    </tr>
                </thead>
                <tbody>
                    <?php
                    $nutrients = [
                        ['key' => 'energy_kj', 'name' => 'Energy', 'unit' => 'kJ', 'ri' => 8400],
                        ['key' => 'energy_kcal', 'name' => '', 'unit' => 'kcal', 'ri' => 2000],
                        ['key' => 'fat', 'name' => 'Fat', 'unit' => 'g', 'ri' => 70],
                        ['key' => 'saturated_fat', 'name' => '  of which saturates', 'unit' => 'g', 'ri' => 20],
                        ['key' => 'carbs', 'name' => 'Carbohydrate', 'unit' => 'g', 'ri' => 260],
                        ['key' => 'sugar', 'name' => '  of which sugars', 'unit' => 'g', 'ri' => 90],
                        ['key' => 'fiber', 'name' => 'Fibre', 'unit' => 'g', 'ri' => null],
                        ['key' => 'protein', 'name' => 'Protein', 'unit' => 'g', 'ri' => 50],
                        ['key' => 'salt', 'name' => 'Salt', 'unit' => 'g', 'ri' => 6],
                    ];

                    foreach ($nutrients as $nutrient):
                        $value = isset($data[$nutrient['key']]) ? $data[$nutrient['key']] : '';
                        if ($value === '' && $nutrient['key'] !== 'energy_kcal') continue;

                        $ri_percent = '';
                        if ($nutrient['ri'] && $value > 0) {
                            $ri_percent = round(($value / $nutrient['ri']) * 100) . '%';
                        }
                    ?>
                    <tr>
                        <td><?php echo esc_html($nutrient['name']); ?></td>
                        <td class="value"><?php echo $value !== '' ? esc_html($value . ' ' . $nutrient['unit']) : '-'; ?></td>
                        <?php if ($serving_size): ?>
                        <td class="value">-</td>
                        <?php endif; ?>
                        <td class="value"><?php echo esc_html($ri_percent); ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>

            <div class="dv-footnote">
                <p>*Reference intake of an average adult (8400 kJ / 2000 kcal)</p>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    
    private static function render_traffic_light_label($data, $options)
    {
        
        $traffic_lights = [];

        
        $fat = floatval($data['fat'] ?? 0);
        $traffic_lights['fat'] = [
            'value' => $fat,
            'unit' => 'g',
            'label' => 'Fat',
            'color' => $fat <= 3 ? 'green' : ($fat <= 17.5 ? 'amber' : 'red')
        ];

        
        $sat = floatval($data['saturated_fat'] ?? 0);
        $traffic_lights['saturates'] = [
            'value' => $sat,
            'unit' => 'g',
            'label' => 'Saturates',
            'color' => $sat <= 1.5 ? 'green' : ($sat <= 5 ? 'amber' : 'red')
        ];

        
        $sugar = floatval($data['sugar'] ?? 0);
        $traffic_lights['sugars'] = [
            'value' => $sugar,
            'unit' => 'g',
            'label' => 'Sugars',
            'color' => $sugar <= 5 ? 'green' : ($sugar <= 22.5 ? 'amber' : 'red')
        ];

        
        $salt = floatval($data['salt'] ?? 0);
        $traffic_lights['salt'] = [
            'value' => $salt,
            'unit' => 'g',
            'label' => 'Salt',
            'color' => $salt <= 0.3 ? 'green' : ($salt <= 1.5 ? 'amber' : 'red')
        ];

        ob_start();
        ?>
        <div class="woonutri-pdf-label traffic-light-style">
            <div class="traffic-lights-container">
                <?php foreach ($traffic_lights as $key => $item): ?>
                <div class="traffic-light-item <?php echo esc_attr($item['color']); ?>">
                    <div class="traffic-light-indicator"></div>
                    <div class="traffic-light-label"><?php echo esc_html($item['label']); ?></div>
                    <div class="traffic-light-value"><?php echo esc_html($item['value'] . $item['unit']); ?></div>
                </div>
                <?php endforeach; ?>
            </div>

            <?php
            
            $calories = isset($data['energy_kcal']) ? $data['energy_kcal'] : 0;
            ?>
            <div class="calories-box" style="text-align: center; margin-top: 8px; padding: 6px; background: #f0f0f0; border-radius: 4px;">
                <div style="font-weight: bold; font-size: 14pt;">Energy</div>
                <div style="font-size: 18pt; font-weight: bold;"><?php echo esc_html($calories); ?> kcal</div>
            </div>

            <?php
            
            echo self::render_standard_label($data, $options);
            ?>
        </div>
        <?php
        return ob_get_clean();
    }

    
    private static function get_label_renderer_css()
    {
        $css_file = WOONUTRI_PLUGIN_DIR . 'assets/css/label.css';

        if (file_exists($css_file)) {
            $css = file_get_contents($css_file);

            
            $css = preg_replace('/@import[^;]+;/', '', $css);

            
            $css .= '
                /* PDF-specific overrides */
                .woonutri-label {
                    max-width: 100% !important;
                    width: 100% !important;
                    margin: 0 !important;
                    page-break-inside: avoid;
                }

                /* Disable animations/transitions for PDF */
                .woonutri-label * {
                    animation: none !important;
                    transition: none !important;
                }

                /* Ensure expandable sections are fully visible in PDF */
                .woonutri-expandable,
                .woonutri-details {
                    display: table-row !important;
                }

                .woonutri-details td {
                    display: table-cell !important;
                }

                /* Force print colors */
                .woonutri-label {
                    -webkit-print-color-adjust: exact !important;
                    print-color-adjust: exact !important;
                }
            ';

            return $css;
        }

        return '';
    }

    
    private static function render_standard_label($data, $options)
    {
        $serving_size = isset($data['serving_size']) ? $data['serving_size'] : '';
        $calories = isset($data['energy_kcal']) ? $data['energy_kcal'] : '0';

        ob_start();
        ?>
        <div class="woonutri-pdf-label standard">
            <div class="nutrition-title">Nutrition Facts</div>

            <?php if ($serving_size): ?>
            <div class="serving-info">
                <div class="serving-size-line">
                    <span>Serving Size</span>
                    <span><?php echo esc_html($serving_size); ?></span>
                </div>
            </div>
            <?php endif; ?>

            <div class="nutrient-row thick-bottom">
                <span class="nutrient-name bold">Calories</span>
                <span class="nutrient-value" style="font-size: 14pt; font-weight: bold;"><?php echo esc_html($calories); ?></span>
            </div>

            <?php
            $nutrients = [
                ['key' => 'fat', 'name' => 'Total Fat', 'unit' => 'g', 'bold' => true],
                ['key' => 'saturated_fat', 'name' => 'Saturated Fat', 'unit' => 'g', 'indent' => true],
                ['key' => 'trans_fat', 'name' => 'Trans Fat', 'unit' => 'g', 'indent' => true],
                ['key' => 'cholesterol', 'name' => 'Cholesterol', 'unit' => 'mg', 'bold' => true],
                ['key' => 'sodium', 'name' => 'Sodium', 'unit' => 'mg', 'bold' => true],
                ['key' => 'carbs', 'name' => 'Total Carbohydrate', 'unit' => 'g', 'bold' => true],
                ['key' => 'fiber', 'name' => 'Dietary Fiber', 'unit' => 'g', 'indent' => true],
                ['key' => 'sugar', 'name' => 'Sugars', 'unit' => 'g', 'indent' => true],
                ['key' => 'protein', 'name' => 'Protein', 'unit' => 'g', 'bold' => true],
                ['key' => 'salt', 'name' => 'Salt', 'unit' => 'g', 'bold' => true],
            ];

            foreach ($nutrients as $nutrient):
                $value = isset($data[$nutrient['key']]) ? $data[$nutrient['key']] : '';
                if ($value === '') continue;

                $classes = ['nutrient-row'];
                if (!empty($nutrient['bold'])) $classes[] = 'bold';
            ?>
            <div class="<?php echo esc_attr(implode(' ', $classes)); ?>">
                <span class="nutrient-name <?php echo !empty($nutrient['indent']) ? 'indent' : ''; ?>">
                    <?php echo esc_html($nutrient['name']); ?>
                </span>
                <span class="nutrient-value"><?php echo esc_html($value . $nutrient['unit']); ?></span>
            </div>
            <?php endforeach; ?>
        </div>
        <?php
        return ob_get_clean();
    }

    
    private static function render_nutri_score($score)
    {
        $score = strtoupper($score);
        if (!in_array($score, ['A', 'B', 'C', 'D', 'E'])) {
            return '';
        }

        ob_start();
        ?>
        <div class="nutri-score-container">
            <div class="nutri-score-bar">
                <?php foreach (['A', 'B', 'C', 'D', 'E'] as $letter): ?>
                <div class="nutri-score-letter score-<?php echo strtolower($letter); ?> <?php echo $letter === $score ? 'active' : ''; ?>">
                    <?php echo $letter; ?>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    
    private static function render_allergens($allergens)
    {
        if (empty($allergens)) {
            return '';
        }

        $allergen_list = is_array($allergens) ? implode(', ', $allergens) : $allergens;

        ob_start();
        ?>
        <div class="allergen-section">
            <span class="allergen-title"><?php esc_html_e('Contains:', 'smart-nutrition-labels'); ?></span>
            <span class="allergen-list"><?php echo esc_html(strtoupper($allergen_list)); ?></span>
        </div>
        <?php
        return ob_get_clean();
    }

    
    private static function render_ingredients($ingredients, $allergens = [])
    {
        if (empty($ingredients)) {
            return '';
        }

        
        $formatted = $ingredients;
        if (!empty($allergens)) {
            $allergen_list = is_array($allergens) ? $allergens : explode(',', $allergens);
            foreach ($allergen_list as $allergen) {
                $allergen = trim($allergen);
                $formatted = preg_replace(
                    '/\b(' . preg_quote($allergen, '/') . ')\b/i',
                    '<span class="allergen-bold">$1</span>',
                    $formatted
                );
            }
        }

        ob_start();
        ?>
        <div class="ingredients-section">
            <span class="ingredients-title"><?php esc_html_e('Ingredients:', 'smart-nutrition-labels'); ?></span>
            <span class="ingredients-list"><?php echo wp_kses($formatted, ['span' => ['class' => []]]); ?></span>
        </div>
        <?php
        return ob_get_clean();
    }

    
    private static function render_regulatory_info($product, $options)
    {
        ob_start();
        ?>
        <div class="regulatory-marks">
            <div class="best-before">
                <?php esc_html_e('Best Before: See Package', 'smart-nutrition-labels'); ?>
            </div>
            <div class="batch-code">
                <?php esc_html_e('Batch:', 'smart-nutrition-labels'); ?> ____________
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    
    public static function generate_product_pdf($product_id, $options = [])
    {
        $product = wc_get_product($product_id);
        if (!$product) {
            return false;
        }

        
        $label_html = self::generate_label_html($product_id, $options);
        if (empty($label_html)) {
            return false;
        }

        return self::generate_pdf($label_html, $options);
    }

    
    public static function generate_batch_pdf($product_ids, $options = [])
    {
        if (empty($product_ids)) {
            return false;
        }

        $defaults = [
            'labels_per_page' => 1,
            'size' => 'medium',
            'template' => 'standard',
            'include_page_numbers' => true,
        ];
        $options = wp_parse_args($options, $defaults);

        $all_labels_html = '';
        $label_count = 0;

        foreach ($product_ids as $product_id) {
            $label_html = self::generate_label_html($product_id, $options);
            if (empty($label_html)) {
                continue;
            }

            $label_count++;

            
            if ($label_count > 1 && $options['labels_per_page'] == 1) {
                $all_labels_html .= '<div style="page-break-before: always;"></div>';
            }

            $all_labels_html .= '<div class="label-wrapper" style="margin-bottom: 20px;">';
            $all_labels_html .= $label_html;
            $all_labels_html .= '</div>';
        }

        if (empty($all_labels_html)) {
            return false;
        }

        
        $options['size'] = 'full_page';

        return self::generate_pdf($all_labels_html, $options);
    }

    
    public static function ajax_export_pdf()
    {
        check_ajax_referer('woonutri_pdf_export', 'nonce');

        
        if (!woonutri_is_pro()) {
            wp_send_json_error([
                'message' => __('PDF Export is a Pro feature', 'smart-nutrition-labels'),
                'upgrade' => true,
                'upgrade_url' => admin_url('admin.php?page=woonutri-settings-pricing'),
                'feature' => 'pdf_export',
                'feature_name' => __('Print-Ready PDF Export', 'smart-nutrition-labels'),
                'feature_description' => __('Export professional nutrition labels as print-ready PDFs for your product packaging.', 'smart-nutrition-labels'),
            ]);
            return;
        }

        $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
        if (!$product_id) {
            wp_send_json_error(['message' => __('Invalid product ID', 'smart-nutrition-labels')]);
            return;
        }

        
        if (!self::is_dompdf_available()) {
            wp_send_json_error([
                'message' => __('PDF library not installed. Please install DomPDF via Composer.', 'smart-nutrition-labels'),
                'install_instructions' => 'Run: composer require dompdf/dompdf'
            ]);
            return;
        }

        
        $options = [
            'size' => isset($_POST['size']) ? sanitize_text_field($_POST['size']) : 'medium',
            'template' => isset($_POST['template']) ? sanitize_text_field($_POST['template']) : 'standard',
            'region' => isset($_POST['region']) ? sanitize_text_field($_POST['region']) : 'us',
            'dpi' => isset($_POST['dpi']) ? intval($_POST['dpi']) : 300,
            'include_bleed' => isset($_POST['include_bleed']) ? (bool) $_POST['include_bleed'] : false,
            'bleed_mm' => isset($_POST['bleed_mm']) ? floatval($_POST['bleed_mm']) : 3,
            'include_crop_marks' => isset($_POST['include_crop_marks']) ? (bool) $_POST['include_crop_marks'] : false,
            'include_product_header' => isset($_POST['include_product_header']) ? (bool) $_POST['include_product_header'] : false,
            'include_allergens' => isset($_POST['include_allergens']) ? (bool) $_POST['include_allergens'] : true,
            'include_ingredients' => isset($_POST['include_ingredients']) ? (bool) $_POST['include_ingredients'] : true,
            'include_nutri_score' => isset($_POST['include_nutri_score']) ? (bool) $_POST['include_nutri_score'] : false,
            'include_traffic_lights' => isset($_POST['include_traffic_lights']) ? (bool) $_POST['include_traffic_lights'] : false,
            'include_barcode_area' => isset($_POST['include_barcode_area']) ? (bool) $_POST['include_barcode_area'] : false,
            'include_regulatory' => isset($_POST['include_regulatory']) ? (bool) $_POST['include_regulatory'] : false,
        ];

        
        if ($options['size'] === 'custom') {
            $options['custom_width'] = isset($_POST['custom_width']) ? floatval($_POST['custom_width']) : 70;
            $options['custom_height'] = isset($_POST['custom_height']) ? floatval($_POST['custom_height']) : 100;
        }

        $pdf_content = self::generate_product_pdf($product_id, $options);

        if (!$pdf_content) {
            wp_send_json_error(['message' => __('Failed to generate PDF. Product may not have nutrition data.', 'smart-nutrition-labels')]);
            return;
        }

        
        $upload_dir = wp_upload_dir();
        $product = wc_get_product($product_id);
        $product_slug = $product ? sanitize_title($product->get_name()) : 'product';
        $filename = 'nutrition-label-' . $product_slug . '-' . time() . '.pdf';
        $filepath = $upload_dir['basedir'] . '/woonutri/' . $filename;
        $fileurl = $upload_dir['baseurl'] . '/woonutri/' . $filename;

        
        $dir_path = $upload_dir['basedir'] . '/woonutri';
        if (!wp_mkdir_p($dir_path)) {
            wp_send_json_error([
                'message' => __('Failed to create PDF directory. Please check file permissions.', 'smart-nutrition-labels'),
            ]);
            return;
        }

        
        $bytes_written = file_put_contents($filepath, $pdf_content);
        if ($bytes_written === false) {
            wp_send_json_error([
                'message' => __('Failed to save PDF file. Please check file permissions.', 'smart-nutrition-labels'),
            ]);
            return;
        }

        
        wp_schedule_single_event(time() + 3600, 'woonutri_cleanup_pdf', [$filepath]);

        wp_send_json_success([
            'url' => $fileurl,
            'filename' => $filename,
            'size' => $bytes_written,
        ]);
    }

    
    public static function ajax_export_pdf_batch()
    {
        check_ajax_referer('woonutri_pdf_export', 'nonce');

        
        if (!woonutri_is_pro()) {
            wp_send_json_error([
                'message' => __('Batch PDF Export is a Pro feature', 'smart-nutrition-labels'),
                'upgrade' => true,
                'upgrade_url' => admin_url('admin.php?page=woonutri-settings-pricing'),
                'feature' => 'pdf_export',
                'feature_name' => __('Batch PDF Export', 'smart-nutrition-labels'),
                'feature_description' => __('Export multiple nutrition labels at once as a single print-ready PDF.', 'smart-nutrition-labels'),
            ]);
            return;
        }

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(['message' => __('Permission denied', 'smart-nutrition-labels')]);
            return;
        }

        $product_ids = isset($_POST['product_ids']) ? array_map('intval', $_POST['product_ids']) : [];
        if (empty($product_ids)) {
            wp_send_json_error(['message' => __('No products selected', 'smart-nutrition-labels')]);
            return;
        }

        if (!self::is_dompdf_available()) {
            wp_send_json_error([
                'message' => __('PDF library not installed', 'smart-nutrition-labels'),
            ]);
            return;
        }

        $options = [
            'size' => isset($_POST['size']) ? sanitize_text_field($_POST['size']) : 'medium',
            'template' => isset($_POST['template']) ? sanitize_text_field($_POST['template']) : 'standard',
            'labels_per_page' => isset($_POST['labels_per_page']) ? intval($_POST['labels_per_page']) : 1,
            'include_product_header' => true,
        ];

        $pdf_content = self::generate_batch_pdf($product_ids, $options);

        if (!$pdf_content) {
            wp_send_json_error(['message' => __('Failed to generate batch PDF', 'smart-nutrition-labels')]);
            return;
        }

        
        $upload_dir = wp_upload_dir();
        $filename = 'nutrition-labels-batch-' . date('Y-m-d-His') . '.pdf';
        $filepath = $upload_dir['basedir'] . '/woonutri/' . $filename;
        $fileurl = $upload_dir['baseurl'] . '/woonutri/' . $filename;

        $dir_path = $upload_dir['basedir'] . '/woonutri';
        if (!wp_mkdir_p($dir_path)) {
            wp_send_json_error([
                'message' => __('Failed to create PDF directory. Please check file permissions.', 'smart-nutrition-labels'),
            ]);
            return;
        }

        $bytes_written = file_put_contents($filepath, $pdf_content);
        if ($bytes_written === false) {
            wp_send_json_error([
                'message' => __('Failed to save PDF file. Please check file permissions.', 'smart-nutrition-labels'),
            ]);
            return;
        }

        wp_schedule_single_event(time() + 3600, 'woonutri_cleanup_pdf', [$filepath]);

        wp_send_json_success([
            'url' => $fileurl,
            'filename' => $filename,
            'count' => count($product_ids),
        ]);
    }

    
    public static function ajax_preview_pdf()
    {
        check_ajax_referer('woonutri_pdf_export', 'nonce');

        if (!function_exists('woonutri_fs') || !woonutri_fs()->can_use_premium_code()) {
            wp_send_json_error(['message' => __('PDF Export is a Pro feature', 'smart-nutrition-labels')]);
            return;
        }

        $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
        if (!$product_id) {
            wp_send_json_error(['message' => __('Invalid product ID', 'smart-nutrition-labels')]);
            return;
        }

        $options = [
            'template' => isset($_POST['template']) ? sanitize_text_field($_POST['template']) : 'standard',
            'include_allergens' => isset($_POST['include_allergens']) ? (bool) $_POST['include_allergens'] : true,
            'include_ingredients' => isset($_POST['include_ingredients']) ? (bool) $_POST['include_ingredients'] : true,
            'include_nutri_score' => isset($_POST['include_nutri_score']) ? (bool) $_POST['include_nutri_score'] : false,
        ];

        $html = self::generate_label_html($product_id, $options);

        if (empty($html)) {
            wp_send_json_error(['message' => __('No nutrition data for this product', 'smart-nutrition-labels')]);
            return;
        }

        wp_send_json_success(['html' => $html]);
    }

    
    public static function ajax_export_pdf_public()
    {
        check_ajax_referer('woonutri_pdf_export', 'nonce');

        
        $ip = $_SERVER['REMOTE_ADDR'];
        $rate_key = 'woonutri_pdf_rate_' . md5($ip);
        $rate_count = get_transient($rate_key);

        if ($rate_count && $rate_count >= 10) {
            wp_send_json_error(['message' => __('Too many requests. Please try again later.', 'smart-nutrition-labels')]);
            return;
        }

        set_transient($rate_key, ($rate_count ?: 0) + 1, HOUR_IN_SECONDS);

        
        self::ajax_export_pdf();
    }

    
    public static function render_export_page()
    {
        
        if (!woonutri_is_pro()) {
            self::render_export_page_with_upsell();
            return;
        }

        
        global $wpdb;
        $products = $wpdb->get_results("
            SELECT DISTINCT p.ID, p.post_title
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            WHERE p.post_type = 'product'
            AND p.post_status = 'publish'
            AND pm.meta_key = 'woonutri_energy_kcal'
            AND pm.meta_value != ''
            ORDER BY p.post_title ASC
            LIMIT 500
        ");

        $dompdf_available = self::is_dompdf_available();
        ?>
        <div class="wrap woonutri-export-wrap">
            <h1><?php esc_html_e('Label Export', 'smart-nutrition-labels'); ?></h1>
            <p class="description"><?php esc_html_e('Generate print-ready PDF nutrition labels for your products.', 'smart-nutrition-labels'); ?></p>

            <?php if (!$dompdf_available): ?>
            <div class="notice notice-error">
                <p>
                    <strong><?php esc_html_e('PDF Library Required', 'smart-nutrition-labels'); ?></strong><br>
                    <?php esc_html_e('DomPDF is not installed. To enable PDF export, run this command in your plugin directory:', 'smart-nutrition-labels'); ?>
                    <code>composer require dompdf/dompdf</code>
                </p>
            </div>
            <?php endif; ?>

            <div class="woonutri-export-container">
                <!-- Left: Options -->
                <div class="woonutri-export-options">
                    <div class="woonutri-export-section">
                        <h3><?php esc_html_e('Select Products', 'smart-nutrition-labels'); ?></h3>
                        <div class="product-selection">
                            <label>
                                <input type="checkbox" id="select-all-products" />
                                <?php esc_html_e('Select All', 'smart-nutrition-labels'); ?>
                            </label>
                            <div class="product-list" style="max-height: 300px; overflow-y: auto; border: 1px solid #ddd; padding: 10px; margin-top: 10px;">
                                <?php if (!empty($products)): ?>
                                    <?php foreach ($products as $product): ?>
                                    <label style="display: block; margin: 5px 0;">
                                        <input type="checkbox" name="product_ids[]" value="<?php echo esc_attr($product->ID); ?>" class="product-checkbox" />
                                        <?php echo esc_html($product->post_title); ?>
                                    </label>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <p><?php esc_html_e('No products with nutrition data found.', 'smart-nutrition-labels'); ?></p>
                                <?php endif; ?>
                            </div>
                            <p class="selected-count" style="margin-top: 10px;">
                                <span id="selected-product-count">0</span> <?php esc_html_e('products selected', 'smart-nutrition-labels'); ?>
                            </p>
                        </div>
                    </div>

                    <div class="woonutri-export-section">
                        <h3><?php esc_html_e('Label Size', 'smart-nutrition-labels'); ?></h3>
                        <select id="label-size" name="size" class="widefat">
                            <?php foreach (self::LABEL_SIZES as $key => $size): ?>
                            <option value="<?php echo esc_attr($key); ?>" <?php selected($key, 'medium'); ?>>
                                <?php echo esc_html($size['name']); ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                        <p class="description size-description"></p>

                        <div id="custom-size-fields" style="display: none; margin-top: 10px;">
                            <label>
                                <?php esc_html_e('Width (mm):', 'smart-nutrition-labels'); ?>
                                <input type="number" id="custom-width" value="70" min="20" max="300" style="width: 80px;" />
                            </label>
                            <label style="margin-left: 10px;">
                                <?php esc_html_e('Height (mm):', 'smart-nutrition-labels'); ?>
                                <input type="number" id="custom-height" value="100" min="20" max="400" style="width: 80px;" />
                            </label>
                        </div>
                    </div>

                    <div class="woonutri-export-section">
                        <h3><?php esc_html_e('Label Template', 'smart-nutrition-labels'); ?></h3>
                        <select id="label-template" name="template" class="widefat">
                            <?php
                            $all_themes = self::get_all_themes();

                            
                            $groups = [
                                'pdf' => ['label' => __('PDF Templates', 'smart-nutrition-labels'), 'themes' => []],
                                'free' => ['label' => __('Free Themes', 'smart-nutrition-labels'), 'themes' => []],
                                'regional' => ['label' => __('Regional Formats', 'smart-nutrition-labels'), 'themes' => []],
                                'design' => ['label' => __('Design Styles', 'smart-nutrition-labels'), 'themes' => []],
                                'specialty' => ['label' => __('Specialty Themes', 'smart-nutrition-labels'), 'themes' => []],
                                'creative' => ['label' => __('Creative Themes', 'smart-nutrition-labels'), 'themes' => []],
                            ];

                            foreach ($all_themes as $key => $theme) {
                                $group = isset($theme['type']) && $theme['type'] === 'pdf' ? 'pdf' : (isset($theme['group']) ? $theme['group'] : 'free');
                                if (isset($groups[$group])) {
                                    $groups[$group]['themes'][$key] = $theme;
                                }
                            }

                            foreach ($groups as $group_key => $group):
                                if (empty($group['themes'])) continue;
                            ?>
                            <optgroup label="<?php echo esc_attr($group['label']); ?>">
                                <?php foreach ($group['themes'] as $key => $theme): ?>
                                <option value="<?php echo esc_attr($key); ?>">
                                    <?php echo esc_html($theme['name']); ?> - <?php echo esc_html($theme['description']); ?>
                                </option>
                                <?php endforeach; ?>
                            </optgroup>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="woonutri-export-section">
                        <h3><?php esc_html_e('Print Options', 'smart-nutrition-labels'); ?></h3>
                        <label style="display: block; margin: 8px 0;">
                            <input type="checkbox" id="include-bleed" name="include_bleed" />
                            <?php esc_html_e('Include bleed area (3mm)', 'smart-nutrition-labels'); ?>
                        </label>
                        <label style="display: block; margin: 8px 0;">
                            <input type="checkbox" id="include-crop-marks" name="include_crop_marks" />
                            <?php esc_html_e('Include crop marks', 'smart-nutrition-labels'); ?>
                        </label>
                        <label style="display: block; margin: 8px 0;">
                            <?php esc_html_e('DPI:', 'smart-nutrition-labels'); ?>
                            <select id="dpi" name="dpi" style="width: 100px;">
                                <option value="150">150 (Draft)</option>
                                <option value="300" selected>300 (Standard)</option>
                                <option value="600">600 (High Quality)</option>
                            </select>
                        </label>
                    </div>

                    <div class="woonutri-export-section">
                        <h3><?php esc_html_e('Content Options', 'smart-nutrition-labels'); ?></h3>
                        <label style="display: block; margin: 8px 0;">
                            <input type="checkbox" id="include-product-header" name="include_product_header" />
                            <?php esc_html_e('Include product name header', 'smart-nutrition-labels'); ?>
                        </label>
                        <label style="display: block; margin: 8px 0;">
                            <input type="checkbox" id="include-allergens" name="include_allergens" checked />
                            <?php esc_html_e('Include allergens section', 'smart-nutrition-labels'); ?>
                        </label>
                        <label style="display: block; margin: 8px 0;">
                            <input type="checkbox" id="include-ingredients" name="include_ingredients" checked />
                            <?php esc_html_e('Include ingredients list', 'smart-nutrition-labels'); ?>
                        </label>
                        <label style="display: block; margin: 8px 0;">
                            <input type="checkbox" id="include-nutri-score" name="include_nutri_score" />
                            <?php esc_html_e('Include Nutri-Score', 'smart-nutrition-labels'); ?>
                        </label>
                        <label style="display: block; margin: 8px 0;">
                            <input type="checkbox" id="include-barcode-area" name="include_barcode_area" />
                            <?php esc_html_e('Include barcode placeholder', 'smart-nutrition-labels'); ?>
                        </label>
                        <label style="display: block; margin: 8px 0;">
                            <input type="checkbox" id="include-regulatory" name="include_regulatory" />
                            <?php esc_html_e('Include regulatory fields (batch, best before)', 'smart-nutrition-labels'); ?>
                        </label>
                    </div>

                    <div class="woonutri-export-actions">
                        <button type="button" id="export-single-pdf" class="button button-primary button-hero" <?php echo !$dompdf_available ? 'disabled' : ''; ?>>
                            <span class="dashicons dashicons-pdf" style="margin-top: 5px;"></span>
                            <?php esc_html_e('Export Single Label', 'smart-nutrition-labels'); ?>
                        </button>
                        <button type="button" id="export-batch-pdf" class="button button-secondary" <?php echo !$dompdf_available ? 'disabled' : ''; ?>>
                            <span class="dashicons dashicons-media-document" style="margin-top: 3px;"></span>
                            <?php esc_html_e('Export All Selected', 'smart-nutrition-labels'); ?>
                        </button>
                    </div>

                    <div id="export-progress" style="display: none; margin-top: 15px;">
                        <div class="progress-bar" style="background: #e0e0e0; border-radius: 4px; height: 20px;">
                            <div class="progress-fill" style="background: #0073aa; height: 100%; width: 0%; border-radius: 4px; transition: width 0.3s;"></div>
                        </div>
                        <p class="progress-text" style="margin-top: 5px;"></p>
                    </div>
                </div>

                <!-- Right: Preview -->
                <div class="woonutri-export-preview">
                    <h3><?php esc_html_e('Label Preview', 'smart-nutrition-labels'); ?></h3>
                    <div id="label-preview" style="border: 1px solid #ddd; padding: 20px; min-height: 400px; background: #f9f9f9;">
                        <p style="color: #666; text-align: center;"><?php esc_html_e('Select a product to see preview', 'smart-nutrition-labels'); ?></p>
                    </div>
                </div>
            </div>
        </div>

        <style>
            .woonutri-export-wrap {
                max-width: 1400px;
            }
            .woonutri-export-container {
                display: grid;
                grid-template-columns: 400px 1fr;
                gap: 30px;
                margin-top: 20px;
            }
            .woonutri-export-section {
                background: #fff;
                padding: 15px;
                border: 1px solid #ddd;
                border-radius: 4px;
                margin-bottom: 15px;
            }
            .woonutri-export-section h3 {
                margin: 0 0 10px 0;
                padding-bottom: 8px;
                border-bottom: 1px solid #eee;
            }
            .woonutri-export-actions {
                display: flex;
                gap: 10px;
                flex-wrap: wrap;
            }
            .woonutri-export-preview {
                background: #fff;
                padding: 20px;
                border: 1px solid #ddd;
                border-radius: 4px;
            }
            @media (max-width: 1200px) {
                .woonutri-export-container {
                    grid-template-columns: 1fr;
                }
            }
        </style>
        <?php
    }

    
    private static function render_export_page_with_upsell()
    {
        
        global $wpdb;
        $products = $wpdb->get_results("
            SELECT DISTINCT p.ID, p.post_title
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            WHERE p.post_type = 'product'
            AND p.post_status = 'publish'
            AND pm.meta_key = 'woonutri_energy_kcal'
            AND pm.meta_value != ''
            ORDER BY p.post_title ASC
            LIMIT 50
        ");
        ?>
        <div class="wrap woonutri-export-page">
            <h1><?php esc_html_e('Nutrition Label Export', 'smart-nutrition-labels'); ?></h1>

            <!-- Upgrade Banner -->
            <div class="woonutri-upgrade-banner" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 25px 30px; border-radius: 8px; margin-bottom: 25px; display: flex; align-items: center; justify-content: space-between; flex-wrap: wrap; gap: 20px;">
                <div>
                    <h2 style="margin: 0 0 8px 0; color: white; font-size: 22px;">
                        <span class="dashicons dashicons-lock" style="font-size: 24px; margin-right: 8px;"></span>
                        <?php esc_html_e('Pro Feature: Print-Ready PDF Export', 'smart-nutrition-labels'); ?>
                    </h2>
                    <p style="margin: 0; opacity: 0.9; font-size: 14px;">
                        <?php esc_html_e('Upgrade to export professional nutrition labels for your product packaging.', 'smart-nutrition-labels'); ?>
                    </p>
                </div>
                <a href="<?php echo esc_url(admin_url('admin.php?page=woonutri-settings-pricing')); ?>" class="button button-hero" style="background: white; color: #667eea; border: none; font-weight: 600; padding: 12px 30px;">
                    <?php esc_html_e('Upgrade to Pro', 'smart-nutrition-labels'); ?>
                </a>
            </div>

            <div class="woonutri-export-container" style="position: relative;">
                <!-- Blur overlay for free users -->
                <div class="woonutri-pro-overlay" style="position: absolute; top: 0; left: 0; right: 0; bottom: 0; background: rgba(255,255,255,0.7); backdrop-filter: blur(3px); z-index: 10; display: flex; align-items: center; justify-content: center; border-radius: 8px;">
                    <div style="text-align: center; padding: 40px; background: white; border-radius: 12px; box-shadow: 0 4px 20px rgba(0,0,0,0.15); max-width: 500px;">
                        <div style="font-size: 56px; margin-bottom: 15px;">🔒</div>
                        <h3 style="margin: 0 0 15px 0; font-size: 24px;"><?php esc_html_e('Unlock PDF Export', 'smart-nutrition-labels'); ?></h3>
                        <ul style="text-align: left; margin: 20px 0; padding-left: 20px;">
                            <li><?php esc_html_e('Multiple label sizes (FDA, EU, custom)', 'smart-nutrition-labels'); ?></li>
                            <li><?php esc_html_e('Professional templates', 'smart-nutrition-labels'); ?></li>
                            <li><?php esc_html_e('High-resolution PDF (up to 600 DPI)', 'smart-nutrition-labels'); ?></li>
                            <li><?php esc_html_e('Bleed areas & crop marks', 'smart-nutrition-labels'); ?></li>
                            <li><?php esc_html_e('Batch export multiple labels', 'smart-nutrition-labels'); ?></li>
                        </ul>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=woonutri-settings-pricing')); ?>" class="button button-primary button-hero">
                            <?php esc_html_e('Upgrade Now', 'smart-nutrition-labels'); ?>
                        </a>
                    </div>
                </div>

                <!-- Actual UI (visible but not interactive for free users) -->
                <div class="woonutri-export-grid" style="display: grid; grid-template-columns: 1fr 350px; gap: 20px; opacity: 0.6; pointer-events: none;">
                    <!-- Left Column: Product Selection -->
                    <div class="woonutri-export-section">
                        <div class="woonutri-card" style="background: white; padding: 20px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
                            <h2><?php esc_html_e('Select Products', 'smart-nutrition-labels'); ?></h2>
                            <p class="description"><?php esc_html_e('Choose products with nutrition data to export.', 'smart-nutrition-labels'); ?></p>

                            <div style="margin: 15px 0;">
                                <label>
                                    <input type="checkbox" id="select-all-products" />
                                    <?php esc_html_e('Select All', 'smart-nutrition-labels'); ?>
                                </label>
                                <span style="margin-left: 15px; color: #666;">
                                    <?php printf(__('Selected: %s', 'smart-nutrition-labels'), '<strong id="selected-product-count">0</strong>'); ?>
                                </span>
                            </div>

                            <div class="woonutri-product-list" style="max-height: 400px; overflow-y: auto; border: 1px solid #ddd; border-radius: 4px;">
                                <?php if (empty($products)): ?>
                                    <p style="padding: 20px; text-align: center; color: #666;">
                                        <?php esc_html_e('No products with nutrition data found.', 'smart-nutrition-labels'); ?>
                                    </p>
                                <?php else: ?>
                                    <?php foreach ($products as $product): ?>
                                        <label style="display: flex; align-items: center; padding: 10px 15px; border-bottom: 1px solid #eee; cursor: pointer;">
                                            <input type="checkbox" class="product-checkbox" value="<?php echo esc_attr($product->ID); ?>" />
                                            <span style="margin-left: 10px;"><?php echo esc_html($product->post_title); ?></span>
                                        </label>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>

                        <!-- Export Options -->
                        <div class="woonutri-card" style="background: white; padding: 20px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1); margin-top: 20px;">
                            <h2><?php esc_html_e('Export Options', 'smart-nutrition-labels'); ?></h2>
                            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
                                <div>
                                    <label for="label-size"><strong><?php esc_html_e('Label Size', 'smart-nutrition-labels'); ?></strong></label>
                                    <select id="label-size" style="width: 100%; margin-top: 5px;">
                                        <?php foreach (self::LABEL_SIZES as $key => $size): ?>
                                            <option value="<?php echo esc_attr($key); ?>"><?php echo esc_html($size['name']); ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div>
                                    <label for="label-template"><strong><?php esc_html_e('Template', 'smart-nutrition-labels'); ?></strong></label>
                                    <select id="label-template" style="width: 100%; margin-top: 5px;">
                                        <?php
                                        $all_themes = self::get_all_themes();
                                        $groups = [
                                            'pdf' => ['label' => __('PDF Templates', 'smart-nutrition-labels'), 'themes' => []],
                                            'free' => ['label' => __('Free Themes', 'smart-nutrition-labels'), 'themes' => []],
                                            'regional' => ['label' => __('Regional', 'smart-nutrition-labels'), 'themes' => []],
                                            'design' => ['label' => __('Design', 'smart-nutrition-labels'), 'themes' => []],
                                            'specialty' => ['label' => __('Specialty', 'smart-nutrition-labels'), 'themes' => []],
                                            'creative' => ['label' => __('Creative', 'smart-nutrition-labels'), 'themes' => []],
                                        ];
                                        foreach ($all_themes as $key => $theme) {
                                            $group = isset($theme['type']) && $theme['type'] === 'pdf' ? 'pdf' : (isset($theme['group']) ? $theme['group'] : 'free');
                                            if (isset($groups[$group])) {
                                                $groups[$group]['themes'][$key] = $theme;
                                            }
                                        }
                                        foreach ($groups as $group_key => $group):
                                            if (empty($group['themes'])) continue;
                                        ?>
                                        <optgroup label="<?php echo esc_attr($group['label']); ?>">
                                            <?php foreach ($group['themes'] as $key => $theme): ?>
                                            <option value="<?php echo esc_attr($key); ?>"><?php echo esc_html($theme['name']); ?></option>
                                            <?php endforeach; ?>
                                        </optgroup>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Right Column: Preview -->
                    <div class="woonutri-export-section">
                        <div class="woonutri-card" style="background: white; padding: 20px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
                            <h2><?php esc_html_e('Preview', 'smart-nutrition-labels'); ?></h2>
                            <div id="label-preview" style="min-height: 300px; background: #f5f5f5; border-radius: 4px; display: flex; align-items: center; justify-content: center;">
                                <p style="color: #666;"><?php esc_html_e('Select a product to see preview', 'smart-nutrition-labels'); ?></p>
                            </div>
                        </div>

                        <!-- Export Buttons -->
                        <div class="woonutri-card" style="background: white; padding: 20px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1); margin-top: 20px;">
                            <button type="button" id="export-single-pdf" class="button button-primary button-large" style="width: 100%;">
                                <?php esc_html_e('Export Single PDF', 'smart-nutrition-labels'); ?>
                            </button>
                            <button type="button" id="export-batch-pdf" class="button button-large" style="width: 100%; margin-top: 10px;">
                                <?php esc_html_e('Export All Selected', 'smart-nutrition-labels'); ?>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    
    private static function render_pro_upsell()
    {
        ?>
        <div class="wrap">
            <div class="woonutri-pro-upsell-card" style="max-width: 600px; margin: 50px auto; text-align: center; padding: 40px; background: #fff; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1);">
                <div style="font-size: 48px; margin-bottom: 20px;">📄</div>
                <h2><?php esc_html_e('Print-Ready Label Export - Pro Feature', 'smart-nutrition-labels'); ?></h2>
                <p><?php esc_html_e('Generate professional, print-ready nutrition labels for your product packaging.', 'smart-nutrition-labels'); ?></p>
                <ul style="text-align: left; max-width: 400px; margin: 20px auto; list-style: none; padding: 0;">
                    <li>✓ <?php esc_html_e('Multiple label sizes (FDA, EU, custom)', 'smart-nutrition-labels'); ?></li>
                    <li>✓ <?php esc_html_e('Professional templates (FDA 2020, EU Table, UK Traffic Light)', 'smart-nutrition-labels'); ?></li>
                    <li>✓ <?php esc_html_e('High-resolution PDF export (up to 600 DPI)', 'smart-nutrition-labels'); ?></li>
                    <li>✓ <?php esc_html_e('Bleed areas and crop marks for professional printing', 'smart-nutrition-labels'); ?></li>
                    <li>✓ <?php esc_html_e('Batch export multiple labels at once', 'smart-nutrition-labels'); ?></li>
                    <li>✓ <?php esc_html_e('Include allergens, ingredients, Nutri-Score', 'smart-nutrition-labels'); ?></li>
                </ul>
                <a href="<?php echo esc_url(woonutri_fs()->get_upgrade_url()); ?>" class="button button-primary button-hero">
                    <?php esc_html_e('Upgrade to Pro', 'smart-nutrition-labels'); ?>
                </a>
            </div>
        </div>
        <?php
    }

    
    public static function render_export_button($product_id, $settings = [])
    {
        $options = get_option('woonutri_options', []);
        $show_pdf_button = isset($options['show_pdf_button']) ? $options['show_pdf_button'] : false;

        if (!$show_pdf_button) {
            return;
        }

        if (!function_exists('woonutri_fs') || !woonutri_fs()->can_use_premium_code()) {
            return;
        }

        $nonce = wp_create_nonce('woonutri_pdf_export');
        ?>
        <div class="woonutri-pdf-export" style="margin-top: 15px; text-align: center;">
            <button type="button" class="woonutri-export-pdf-btn"
                    data-product-id="<?php echo esc_attr($product_id); ?>"
                    data-nonce="<?php echo esc_attr($nonce); ?>"
                    title="<?php esc_attr_e('Download label as PDF', 'smart-nutrition-labels'); ?>"
                    style="display: inline-flex; align-items: center; gap: 5px; padding: 8px 16px; background: #c62828; color: white; border: none; border-radius: 4px; cursor: pointer; font-size: 13px;">
                <span class="dashicons dashicons-pdf" style="font-size: 16px; width: 16px; height: 16px;"></span>
                <?php esc_html_e('Download PDF', 'smart-nutrition-labels'); ?>
            </button>
        </div>
        <?php
    }

    
    public static function render_admin_export_panel($product_id)
    {
        if (!function_exists('woonutri_fs') || !woonutri_fs()->can_use_premium_code()) {
            return;
        }

        $nonce = wp_create_nonce('woonutri_pdf_export');
        ?>
        <div class="woonutri-admin-pdf-export" style="margin-top: 20px; padding: 15px; background: #f8f9fa; border-radius: 6px; border: 1px solid #ddd;">
            <h4 style="margin: 0 0 10px 0;">
                <span class="dashicons dashicons-pdf" style="color: #c62828;"></span>
                <?php esc_html_e('Label Export', 'smart-nutrition-labels'); ?>
            </h4>

            <div style="display: flex; gap: 10px; flex-wrap: wrap; margin-bottom: 10px;">
                <select id="admin-export-size" style="min-width: 150px;">
                    <option value="small"><?php esc_html_e('Small (50x70mm)', 'smart-nutrition-labels'); ?></option>
                    <option value="medium" selected><?php esc_html_e('Medium (70x100mm)', 'smart-nutrition-labels'); ?></option>
                    <option value="large"><?php esc_html_e('Large (100x150mm)', 'smart-nutrition-labels'); ?></option>
                    <option value="fda_standard"><?php esc_html_e('FDA Standard', 'smart-nutrition-labels'); ?></option>
                    <option value="eu_standard"><?php esc_html_e('EU Standard', 'smart-nutrition-labels'); ?></option>
                </select>

                <select id="admin-export-template" style="min-width: 150px;">
                    <option value="standard"><?php esc_html_e('Standard', 'smart-nutrition-labels'); ?></option>
                    <option value="fda_2020"><?php esc_html_e('FDA 2020', 'smart-nutrition-labels'); ?></option>
                    <option value="eu_table"><?php esc_html_e('EU Table', 'smart-nutrition-labels'); ?></option>
                    <option value="traffic_light"><?php esc_html_e('UK Traffic Light', 'smart-nutrition-labels'); ?></option>
                </select>
            </div>

            <button type="button" class="button button-primary woonutri-admin-export-pdf"
                    data-product-id="<?php echo esc_attr($product_id); ?>"
                    data-nonce="<?php echo esc_attr($nonce); ?>">
                <span class="dashicons dashicons-download" style="margin-top: 3px;"></span>
                <?php esc_html_e('Export PDF', 'smart-nutrition-labels'); ?>
            </button>

            <a href="<?php echo esc_url(admin_url('admin.php?page=woonutri-export')); ?>" class="button" style="margin-left: 5px;">
                <?php esc_html_e('Advanced Options', 'smart-nutrition-labels'); ?>
            </a>

            <p class="description" style="margin-top: 8px;">
                <?php esc_html_e('Download a print-ready PDF for physical packaging.', 'smart-nutrition-labels'); ?>
            </p>
        </div>
        <?php
    }

    
    public static function cleanup_pdf($filepath)
    {
        if (file_exists($filepath) && is_writable($filepath)) {
            @unlink($filepath);
        }
    }

    
    public static function add_column_styles()
    {
        $screen = get_current_screen();
        if (!$screen || $screen->id !== 'edit-product') {
            return;
        }
        ?>
        <style>
            .column-woonutri_status {
                width: 70px !important;
                text-align: center !important;
            }
            .column-woonutri_status .dashicons-carrot {
                color: #f57c00;
                font-size: 18px;
            }
            .column-woonutri_status .dashicons-yes-alt {
                color: #46b450;
            }
            .column-woonutri_status .dashicons-warning {
                color: #ffb900;
            }
            .column-woonutri_status .dashicons-minus {
                color: #ccc;
            }
            .column-woonutri_status .dashicons-pdf {
                color: #c62828;
                font-size: 14px;
                vertical-align: middle;
            }
            .column-woonutri_status a {
                text-decoration: none;
            }
            .woonutri-nutri-score {
                display: inline-block;
                padding: 1px 5px;
                border-radius: 3px;
                font-size: 10px;
                font-weight: bold;
                color: #fff;
                vertical-align: middle;
                margin-left: 3px;
            }
        </style>
        <?php
    }

    
    public static function get_label_sizes()
    {
        return self::LABEL_SIZES;
    }

    
    public static function get_label_templates()
    {
        return self::LABEL_TEMPLATES;
    }

    
    public static function add_product_row_action($actions, $post)
    {
        
        if ($post->post_type !== 'product') {
            return $actions;
        }

        
        $has_nutrition = get_post_meta($post->ID, 'woonutri_energy_kcal', true);
        if (empty($has_nutrition)) {
            return $actions;
        }

        
        $export_url = wp_nonce_url(
            add_query_arg([
                'action' => 'woonutri_quick_export_pdf',
                'product_id' => $post->ID,
            ], admin_url('admin-post.php')),
            'woonutri_quick_export_' . $post->ID
        );

        
        $actions['woonutri_export_pdf'] = sprintf(
            '<a href="%s" class="woonutri-quick-export" title="%s" target="_blank">%s</a>',
            esc_url($export_url),
            esc_attr__('Export nutrition label as PDF', 'smart-nutrition-labels'),
            esc_html__('Export PDF', 'smart-nutrition-labels')
        );

        return $actions;
    }

    
    public static function add_bulk_export_action($bulk_actions)
    {
        $bulk_actions['woonutri_bulk_export_pdf'] = __('Export Nutrition Labels (PDF)', 'smart-nutrition-labels');
        return $bulk_actions;
    }

    
    public static function handle_bulk_export_action($redirect_to, $doaction, $post_ids)
    {
        if ($doaction !== 'woonutri_bulk_export_pdf') {
            return $redirect_to;
        }

        
        if (!woonutri_is_pro()) {
            return add_query_arg([
                'woonutri_bulk_export' => 'upgrade_required',
                'count' => 0,
            ], $redirect_to);
        }

        
        $valid_ids = [];
        foreach ($post_ids as $post_id) {
            $has_nutrition = get_post_meta($post_id, 'woonutri_energy_kcal', true);
            if (!empty($has_nutrition)) {
                $valid_ids[] = $post_id;
            }
        }

        if (empty($valid_ids)) {
            return add_query_arg([
                'woonutri_bulk_export' => 'no_nutrition',
                'count' => 0,
            ], $redirect_to);
        }

        
        if (!self::is_dompdf_available()) {
            return add_query_arg([
                'woonutri_bulk_export' => 'no_dompdf',
                'count' => 0,
            ], $redirect_to);
        }

        
        $options = [
            'size' => 'medium',
            'template' => 'standard',
            'include_product_header' => true,
            'include_allergens' => true,
            'include_ingredients' => true,
        ];

        $pdf_content = self::generate_batch_pdf($valid_ids, $options);

        if (!$pdf_content) {
            return add_query_arg([
                'woonutri_bulk_export' => 'error',
                'count' => 0,
            ], $redirect_to);
        }

        
        $upload_dir = wp_upload_dir();
        $filename = 'nutrition-labels-batch-' . date('Y-m-d-His') . '.pdf';
        $filepath = $upload_dir['basedir'] . '/woonutri/' . $filename;
        $fileurl = $upload_dir['baseurl'] . '/woonutri/' . $filename;

        $dir_path = $upload_dir['basedir'] . '/woonutri';
        if (!wp_mkdir_p($dir_path)) {
            return add_query_arg([
                'woonutri_bulk_export' => 'error',
                'woonutri_error' => 'directory',
                'count' => 0,
            ], $redirect_to);
        }

        $bytes_written = file_put_contents($filepath, $pdf_content);
        if ($bytes_written === false) {
            return add_query_arg([
                'woonutri_bulk_export' => 'error',
                'woonutri_error' => 'write',
                'count' => 0,
            ], $redirect_to);
        }

        wp_schedule_single_event(time() + 3600, 'woonutri_cleanup_pdf', [$filepath]);

        
        set_transient('woonutri_bulk_export_url_' . get_current_user_id(), $fileurl, 60);

        return add_query_arg([
            'woonutri_bulk_export' => 'success',
            'count' => count($valid_ids),
        ], $redirect_to);
    }

    
    public static function bulk_export_admin_notice()
    {
        if (!isset($_GET['woonutri_bulk_export'])) {
            return;
        }

        $status = sanitize_text_field($_GET['woonutri_bulk_export']);
        $count = isset($_GET['count']) ? intval($_GET['count']) : 0;

        switch ($status) {
            case 'success':
                $url = get_transient('woonutri_bulk_export_url_' . get_current_user_id());
                delete_transient('woonutri_bulk_export_url_' . get_current_user_id());

                if ($url) {
                    printf(
                        '<div class="notice notice-success is-dismissible"><p>%s <a href="%s" target="_blank" class="button button-small" style="margin-left: 10px;">%s</a></p></div>',
                        sprintf(
                            esc_html(_n(
                                'Successfully exported %d nutrition label.',
                                'Successfully exported %d nutrition labels.',
                                $count,
                                'woonutri'
                            )),
                            $count
                        ),
                        esc_url($url),
                        esc_html__('Download PDF', 'smart-nutrition-labels')
                    );
                }
                break;

            case 'no_nutrition':
                echo '<div class="notice notice-warning is-dismissible"><p>' .
                    esc_html__('No products with nutrition data were selected.', 'smart-nutrition-labels') .
                    '</p></div>';
                break;

            case 'no_dompdf':
                echo '<div class="notice notice-error is-dismissible"><p>' .
                    esc_html__('PDF library (DomPDF) is not installed. Please run: composer require dompdf/dompdf', 'smart-nutrition-labels') .
                    '</p></div>';
                break;

            case 'error':
                echo '<div class="notice notice-error is-dismissible"><p>' .
                    esc_html__('Failed to generate PDF. Please try again.', 'smart-nutrition-labels') .
                    '</p></div>';
                break;

            case 'upgrade_required':
                $upgrade_url = admin_url('admin.php?page=woonutri-settings-pricing');
                printf(
                    '<div class="notice notice-info is-dismissible"><p>%s <a href="%s" class="button button-primary button-small" style="margin-left: 10px;">%s</a></p></div>',
                    esc_html__('PDF Export is a Pro feature. Upgrade to export nutrition labels as print-ready PDFs.', 'smart-nutrition-labels'),
                    esc_url($upgrade_url),
                    esc_html__('Upgrade to Pro', 'smart-nutrition-labels')
                );
                break;
        }
    }

    
    public static function add_nutrition_column($columns)
    {
        $new_columns = [];

        foreach ($columns as $key => $label) {
            $new_columns[$key] = $label;

            
            
            if ($key === 'name' || $key === 'product') {
                $new_columns['woonutri_status'] = '<span class="dashicons dashicons-carrot" style="color: #f57c00;" title="' .
                    esc_attr__('Nutrition Data', 'smart-nutrition-labels') . '"></span><span class="screen-reader-text">' .
                    esc_html__('Nutrition', 'smart-nutrition-labels') . '</span>';
            }
        }

        
        if (!isset($new_columns['woonutri_status'])) {
            
            if (isset($columns['date'])) {
                $new_columns = [];
                foreach ($columns as $key => $label) {
                    if ($key === 'date') {
                        $new_columns['woonutri_status'] = '<span class="dashicons dashicons-carrot" style="color: #f57c00;" title="' .
                            esc_attr__('Nutrition Data', 'smart-nutrition-labels') . '"></span><span class="screen-reader-text">' .
                            esc_html__('Nutrition', 'smart-nutrition-labels') . '</span>';
                    }
                    $new_columns[$key] = $label;
                }
            } else {
                
                $new_columns['woonutri_status'] = '<span class="dashicons dashicons-carrot" style="color: #f57c00;" title="' .
                    esc_attr__('Nutrition Data', 'smart-nutrition-labels') . '"></span><span class="screen-reader-text">' .
                    esc_html__('Nutrition', 'smart-nutrition-labels') . '</span>';
            }
        }

        return $new_columns;
    }

    
    public static function render_nutrition_column($column, $post_id)
    {
        if ($column !== 'woonutri_status') {
            return;
        }

        $energy = get_post_meta($post_id, 'woonutri_energy_kcal', true);
        $has_nutrition = !empty($energy);

        if ($has_nutrition) {
            
            $required_fields = ['energy_kcal', 'fat', 'carbs', 'protein'];
            $complete = true;
            foreach ($required_fields as $field) {
                $value = get_post_meta($post_id, 'woonutri_' . $field, true);
                if (empty($value)) {
                    $complete = false;
                    break;
                }
            }

            
            $nutri_score = get_post_meta($post_id, 'woonutri_nutri_score', true);

            if ($complete) {
                $tooltip = sprintf(
                    __('Nutrition data complete (%s kcal)', 'smart-nutrition-labels'),
                    number_format_i18n($energy)
                );
                $icon = '<span class="dashicons dashicons-yes-alt" style="color: #46b450;" title="' . esc_attr($tooltip) . '"></span>';
            } else {
                $tooltip = __('Nutrition data incomplete', 'smart-nutrition-labels');
                $icon = '<span class="dashicons dashicons-warning" style="color: #ffb900;" title="' . esc_attr($tooltip) . '"></span>';
            }

            
            if ($nutri_score) {
                $score_colors = [
                    'A' => '#038141',
                    'B' => '#85BB2F',
                    'C' => '#FECB02',
                    'D' => '#EE8100',
                    'E' => '#E63946',
                ];
                $color = isset($score_colors[strtoupper($nutri_score)]) ? $score_colors[strtoupper($nutri_score)] : '#666';
                $icon .= sprintf(
                    ' <span style="background: %s; color: #fff; padding: 1px 5px; border-radius: 3px; font-size: 10px; font-weight: bold;">%s</span>',
                    $color,
                    strtoupper($nutri_score)
                );
            }

            
            $export_url = wp_nonce_url(
                add_query_arg([
                    'action' => 'woonutri_quick_export_pdf',
                    'product_id' => $post_id,
                ], admin_url('admin-post.php')),
                'woonutri_quick_export_' . $post_id
            );

            echo $icon . ' <a href="' . esc_url($export_url) . '" target="_blank" title="' .
                esc_attr__('Export PDF', 'smart-nutrition-labels') . '" style="color: #c62828;"><span class="dashicons dashicons-pdf" style="font-size: 16px;"></span></a>';
        } else {
            echo '<span class="dashicons dashicons-minus" style="color: #ccc;" title="' .
                esc_attr__('No nutrition data', 'smart-nutrition-labels') . '"></span>';
        }
    }

    
    public static function make_nutrition_column_sortable($columns)
    {
        $columns['woonutri_status'] = 'woonutri_status';
        return $columns;
    }

    
    public static function sort_by_nutrition_column($query)
    {
        if (!is_admin() || !$query->is_main_query()) {
            return;
        }

        if ($query->get('orderby') !== 'woonutri_status') {
            return;
        }

        $query->set('meta_key', 'woonutri_energy_kcal');
        $query->set('orderby', 'meta_value_num');
    }

    
    public static function handle_quick_export()
    {
        $product_id = isset($_GET['product_id']) ? intval($_GET['product_id']) : 0;

        if (!$product_id) {
            wp_die(__('Invalid product ID', 'smart-nutrition-labels'));
        }

        
        if (!wp_verify_nonce($_GET['_wpnonce'], 'woonutri_quick_export_' . $product_id)) {
            wp_die(__('Security check failed', 'smart-nutrition-labels'));
        }

        
        if (!current_user_can('edit_post', $product_id)) {
            wp_die(__('Permission denied', 'smart-nutrition-labels'));
        }

        
        if (!woonutri_is_pro()) {
            wp_redirect(admin_url('admin.php?page=woonutri-settings-pricing&feature=pdf_export'));
            exit;
        }

        
        if (!self::is_dompdf_available()) {
            wp_die(__('PDF library not installed. Please install DomPDF via Composer.', 'smart-nutrition-labels'));
        }

        
        $options = [
            'size' => 'medium',
            'template' => 'standard',
            'include_allergens' => true,
            'include_ingredients' => true,
        ];

        $pdf_content = self::generate_product_pdf($product_id, $options);

        if (!$pdf_content) {
            wp_die(__('Failed to generate PDF. Product may not have nutrition data.', 'smart-nutrition-labels'));
        }

        
        $product = wc_get_product($product_id);
        $product_slug = $product ? sanitize_title($product->get_name()) : 'product';

        
        header('Content-Type: application/pdf');
        header('Content-Disposition: attachment; filename="nutrition-label-' . $product_slug . '.pdf"');
        header('Content-Length: ' . strlen($pdf_content));
        header('Cache-Control: private, max-age=0, must-revalidate');
        header('Pragma: public');

        echo $pdf_content;
        exit;
    }
}
