<?php
class Woonutri_Traffic_Lights
{

    const THRESHOLDS_PER_100G = [
        'fat' => [
            'green' => 3.0,
            'amber' => 17.5,
        ],
        'saturated_fat' => [
            'green' => 1.5,
            'amber' => 5.0,
        ],
        'sugar' => [
            'green' => 5.0,
            'amber' => 22.5,
        ],
        'salt' => [
            'green' => 0.3,
            'amber' => 1.5,
        ],
    ];

    const THRESHOLDS_PER_SERVING = [
        'fat' => [
            'green' => 3.0,
            'amber' => 21.0,
        ],
        'saturated_fat' => [
            'green' => 1.5,
            'amber' => 6.0,
        ],
        'sugar' => [
            'green' => 6.75,
            'amber' => 27.0,
        ],
        'salt' => [
            'green' => 0.3,
            'amber' => 1.8,
        ],
    ];

    const ALLERGEN_ICONS = [
        'milk' => '🥛',
        'dairy' => '🥛',
        'eggs' => '🥚',
        'egg' => '🥚',
        'peanuts' => '🥜',
        'peanut' => '🥜',
        'tree nuts' => '🌰',
        'nuts' => '🌰',
        'almonds' => '🌰',
        'hazelnuts' => '🌰',
        'walnuts' => '🌰',
        'cashews' => '🌰',
        'fish' => '🐟',
        'shellfish' => '🦐',
        'crustaceans' => '🦐',
        'molluscs' => '🦪',
        'wheat' => '🌾',
        'gluten' => '🌾',
        'cereals' => '🌾',
        'barley' => '🌾',
        'rye' => '🌾',
        'oats' => '🌾',
        'soy' => '🫘',
        'soybeans' => '🫘',
        'soya' => '🫘',
        'sesame' => '⚪',
        'celery' => '🥬',
        'mustard' => '🟡',
        'sulphites' => '⚠️',
        'sulphur dioxide' => '⚠️',
        'lupin' => '🌸',
    ];

    public static function get_color($nutrient, $value, $per_serving = false)
    {
        $thresholds = $per_serving ? self::THRESHOLDS_PER_SERVING : self::THRESHOLDS_PER_100G;

        if (!isset($thresholds[$nutrient])) {
            return 'neutral';
        }

        $limits = $thresholds[$nutrient];
        $value = floatval($value);

        if ($value <= $limits['green']) {
            return 'green';
        } elseif ($value <= $limits['amber']) {
            return 'amber';
        } else {
            return 'red';
        }
    }

    public static function get_css_class($color)
    {
        $classes = [
            'green' => 'woonutri-traffic-green',
            'amber' => 'woonutri-traffic-amber',
            'red' => 'woonutri-traffic-red',
            'neutral' => 'woonutri-traffic-neutral',
        ];
        return isset($classes[$color]) ? $classes[$color] : $classes['neutral'];
    }

    public static function render_badge($nutrient, $value, $label = '', $per_serving = false)
    {
        $color = self::get_color($nutrient, $value, $per_serving);
        $css_class = self::get_css_class($color);

        $labels = [
            'green' => __('LOW', 'smart-nutrition-labels'),
            'amber' => __('MED', 'smart-nutrition-labels'),
            'red' => __('HIGH', 'smart-nutrition-labels'),
        ];

        $level_label = isset($labels[$color]) ? $labels[$color] : '';

        $tooltip = self::get_tooltip($nutrient, $color);

        return sprintf(
            '<span class="woonutri-traffic-badge %s" title="%s">%s</span>',
            esc_attr($css_class),
            esc_attr($tooltip),
            esc_html($level_label)
        );
    }

    public static function render_traffic_lights_row($nutrition_data, $per_serving = false)
    {
        $nutrients = [
            'fat' => __('Fat', 'smart-nutrition-labels'),
            'saturated_fat' => __('Saturates', 'smart-nutrition-labels'),
            'sugar' => __('Sugars', 'smart-nutrition-labels'),
            'salt' => __('Salt', 'smart-nutrition-labels'),
        ];

        $output = '<div class="woonutri-traffic-lights-row">';

        foreach ($nutrients as $key => $label) {
            $value = isset($nutrition_data[$key]) ? floatval($nutrition_data[$key]) : 0;
            $color = self::get_color($key, $value, $per_serving);
            $css_class = self::get_css_class($color);

            $output .= sprintf(
                '<div class="woonutri-traffic-item %s" title="%s">
                    <span class="woonutri-traffic-light"></span>
                    <span class="woonutri-traffic-label">%s</span>
                    <span class="woonutri-traffic-value">%sg</span>
                </div>',
                esc_attr($css_class),
                esc_attr(self::get_tooltip($key, $color)),
                esc_html($label),
                esc_html(number_format($value, 1))
            );
        }

        $output .= '</div>';

        return $output;
    }

    private static function get_tooltip($nutrient, $color)
    {
        $nutrient_labels = [
            'fat' => __('fat', 'smart-nutrition-labels'),
            'saturated_fat' => __('saturated fat', 'smart-nutrition-labels'),
            'sugar' => __('sugar', 'smart-nutrition-labels'),
            'salt' => __('salt', 'smart-nutrition-labels'),
        ];

        $color_descriptions = [
            'green' => __('This product is LOW in %s. Great choice!', 'smart-nutrition-labels'),
            'amber' => __('This product is MEDIUM in %s. Okay to have sometimes.', 'smart-nutrition-labels'),
            'red' => __('This product is HIGH in %s. Have occasionally.', 'smart-nutrition-labels'),
        ];

        $label = isset($nutrient_labels[$nutrient]) ? $nutrient_labels[$nutrient] : $nutrient;
        $template = isset($color_descriptions[$color]) ? $color_descriptions[$color] : '%s';

        return sprintf($template, $label);
    }

    public static function get_allergen_icons($allergen_text)
    {
        if (empty($allergen_text)) {
            return [];
        }

        $icons = [];
        $text_lower = strtolower($allergen_text);

        foreach (self::ALLERGEN_ICONS as $allergen => $icon) {
            if (strpos($text_lower, $allergen) !== false) {
                if (!in_array($icon, $icons)) {
                    $icons[$allergen] = $icon;
                }
            }
        }

        return $icons;
    }

    public static function render_allergen_icons($allergen_text)
    {
        $icons = self::get_allergen_icons($allergen_text);

        if (empty($icons)) {
            return '';
        }

        $output = '<div class="woonutri-allergen-icons" title="' . esc_attr__('Contains allergens', 'smart-nutrition-labels') . '">';
        $output .= '<span class="woonutri-allergen-label">' . esc_html__('Contains:', 'smart-nutrition-labels') . '</span> ';

        foreach ($icons as $allergen => $icon) {
            $output .= sprintf(
                '<span class="woonutri-allergen-icon" title="%s">%s</span>',
                esc_attr(ucfirst($allergen)),
                $icon
            );
        }

        $output .= '</div>';

        return $output;
    }

    public static function get_styles()
    {
        return '
        .woonutri-traffic-lights-row {
            display: flex;
            gap: 12px;
            padding: 12px;
            background: #f8f9fa;
            border-radius: 8px;
            margin: 12px 0;
            flex-wrap: wrap;
            justify-content: center;
        }

        .woonutri-traffic-item {
            display: flex;
            flex-direction: column;
            align-items: center;
            padding: 8px 16px;
            border-radius: 6px;
            min-width: 70px;
            cursor: help;
        }

        .woonutri-traffic-item.woonutri-traffic-green {
            background: #d4edda;
            border: 2px solid #28a745;
        }

        .woonutri-traffic-item.woonutri-traffic-amber {
            background: #fff3cd;
            border: 2px solid #ffc107;
        }

        .woonutri-traffic-item.woonutri-traffic-red {
            background: #f8d7da;
            border: 2px solid #dc3545;
        }

        .woonutri-traffic-light {
            width: 20px;
            height: 20px;
            border-radius: 50%;
            margin-bottom: 4px;
        }

        .woonutri-traffic-green .woonutri-traffic-light { background: #28a745; }
        .woonutri-traffic-amber .woonutri-traffic-light { background: #ffc107; }
        .woonutri-traffic-red .woonutri-traffic-light { background: #dc3545; }

        .woonutri-traffic-label {
            font-size: 11px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .woonutri-traffic-value {
            font-size: 13px;
            font-weight: 500;
            margin-top: 2px;
        }

        .woonutri-traffic-badge {
            display: inline-block;
            padding: 2px 6px;
            border-radius: 4px;
            font-size: 10px;
            font-weight: 700;
            text-transform: uppercase;
            margin-left: 6px;
        }

        .woonutri-traffic-badge.woonutri-traffic-green {
            background: #28a745;
            color: white;
        }
        .woonutri-traffic-badge.woonutri-traffic-amber {
            background: #ffc107;
            color: #212529;
        }
        .woonutri-traffic-badge.woonutri-traffic-red {
            background: #dc3545;
            color: white;
        }

        .woonutri-allergen-icons {
            display: flex;
            align-items: center;
            gap: 6px;
            padding: 8px 12px;
            background: #fff3cd;
            border: 1px solid #ffc107;
            border-radius: 6px;
            margin: 8px 0;
            flex-wrap: wrap;
        }

        .woonutri-allergen-label {
            font-weight: 600;
            font-size: 12px;
            color: #856404;
        }

        .woonutri-allergen-icon {
            font-size: 18px;
            cursor: help;
            transition: transform 0.2s;
        }

        .woonutri-allergen-icon:hover {
            transform: scale(1.2);
        }
        ';
    }
}
