<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * @package WooNutri
 * @since 2.0.0
 */

class Woonutri_Label_Renderer
{
    
    private static $instance = null;

    
    public static function get_instance()
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    
    public static function init()
    {
        $instance = self::get_instance();

        
        

        
        if (!is_admin() || wp_doing_ajax()) {
            add_action('wp_enqueue_scripts', [__CLASS__, 'enqueue_assets']);

            
            $instance->init_display_hooks();
        }
    }

    
    private function __construct()
    {
        
    }

    
    public function init_display_hooks()
    {
        
        if (!class_exists('WooCommerce')) {
            return;
        }

        
        remove_action('woocommerce_single_product_summary', [$this, 'auto_render_label'], 25);
        remove_filter('woocommerce_product_tabs', [$this, 'add_nutrition_tab']);

        
        $options = get_option('woonutri_options', []);
        $display_position = isset($options['display_position']) ? $options['display_position'] : 'summary';

        
        switch ($display_position) {
            case 'summary':
                add_action('woocommerce_single_product_summary', [$this, 'auto_render_label'], 25);
                break;

            case 'tab':
                add_filter('woocommerce_product_tabs', [$this, 'add_nutrition_tab']);
                break;

            case 'both':
                add_action('woocommerce_single_product_summary', [$this, 'auto_render_label'], 25);
                add_filter('woocommerce_product_tabs', [$this, 'add_nutrition_tab']);
                break;
        }
    }

    
    public static function enqueue_assets()
    {
        if (is_product() || apply_filters('woonutri_always_enqueue_assets', false)) {
            
            wp_enqueue_style('woonutri-label', WOONUTRI_PLUGIN_URL . 'assets/css/label.css', [], WOONUTRI_VERSION);

            
            $options = get_option('woonutri_options', []);
            $active_theme = isset($options['label_theme']) ? $options['label_theme'] : 'basic';

            
            $fda_themes = ['fda', 'us-fda'];
            $eu_themes = ['eu', 'uk'];

            if (in_array($active_theme, $fda_themes, true)) {
                wp_enqueue_style('woonutri-label-fda', WOONUTRI_PLUGIN_URL . 'assets/css/label-fda.css', ['woonutri-label'], WOONUTRI_VERSION);
            } elseif (in_array($active_theme, $eu_themes, true)) {
                wp_enqueue_style('woonutri-label-eu', WOONUTRI_PLUGIN_URL . 'assets/css/label-eu.css', ['woonutri-label'], WOONUTRI_VERSION);
            }

            
            self::enqueue_google_font($options);

            
            $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();
            if ($is_pro) {
                self::enqueue_pro_styles($options);
            }

            wp_enqueue_script('woonutri-nutrition-label', WOONUTRI_PLUGIN_URL . 'assets/js/nutrition-label.js', ['jquery'], WOONUTRI_VERSION, true);

            
            do_action('woonutri_after_enqueue_assets');
        }
    }

    
    private static function enqueue_google_font($options)
    {
        $custom_font = isset($options['custom_font']) ? $options['custom_font'] : '';

        if (empty($custom_font)) {
            return;
        }

        
        $cache_key = 'google_font_' . md5($custom_font);

        if (class_exists('Woonutri_Cache')) {
            $font_url = Woonutri_Cache::get($cache_key);

            if (false === $font_url) {
                $font_url = 'https://fonts.googleapis.com/css2?family=' . urlencode(str_replace(' ', '+', $custom_font)) . ':wght@400;700&display=swap';
                Woonutri_Cache::set($cache_key, $font_url, WEEK_IN_SECONDS);
            }
        } else {
            
            $font_url = get_transient('woonutri_' . $cache_key);

            if (false === $font_url) {
                $font_url = 'https://fonts.googleapis.com/css2?family=' . urlencode(str_replace(' ', '+', $custom_font)) . ':wght@400;700&display=swap';
                set_transient('woonutri_' . $cache_key, $font_url, WEEK_IN_SECONDS);
            }
        }

        wp_enqueue_style('woonutri-google-font', $font_url, [], null);
    }

    
    private static function enqueue_pro_styles($options)
    {
        $css = '';

        
        if (!empty($options['enable_shadow'])) {
            $intensity = isset($options['shadow_intensity']) ? intval($options['shadow_intensity']) : 5;
            $blur = $intensity * 2;
            $spread = intval($intensity / 2);
            $opacity = min(0.1 + ($intensity * 0.02), 0.4);
            $css .= ".woonutri-label { box-shadow: 0 {$intensity}px {$blur}px {$spread}px rgba(0,0,0,{$opacity}); }\n";
        }

        
        if (!empty($options['enable_gradient']) && !empty($options['gradient_colors'])) {
            $colors = array_map('sanitize_hex_color', explode(',', $options['gradient_colors']));
            if (count($colors) >= 2) {
                $gradient = 'linear-gradient(135deg, ' . implode(', ', $colors) . ')';
                $css .= ".woonutri-label { background: {$gradient}; }\n";
            }
        }

        
        if (!empty($options['enable_animations'])) {
            $css .= ".woonutri-label { transition: transform 0.3s ease, box-shadow 0.3s ease; }\n";
            $css .= ".woonutri-label:hover { transform: translateY(-2px); }\n";
        }

        
        if (!empty($options['custom_css'])) {
            $css .= wp_strip_all_tags($options['custom_css']) . "\n";
        }

        if ($css) {
            wp_add_inline_style('woonutri-label', $css);
        }
    }

    
    public function auto_render_label($args = [])
    {
        $product_id = get_the_ID();
        $position = isset($args['position']) ? $args['position'] : 'default';
        $theme = isset($args['theme']) ? $args['theme'] : '';
        $show_title = isset($args['show_title']) ? (bool) $args['show_title'] : true;

        
        $data = self::get_nutrition_data($product_id);
        if (empty(array_filter($data))) {
            return; 
        }

        try {
            echo self::render_label($product_id, [], [
                'position' => $position,
                'theme' => $theme,
                'show_title' => $show_title
            ]);
        } catch (Exception $e) {
            if (current_user_can('manage_options')) {
                echo '<!-- WooNutri Error: ' . esc_html($e->getMessage()) . ' -->';
            }
        }
    }

    
    public function add_nutrition_tab($tabs)
    {
        global $product;

        if (!$product) {
            return $tabs;
        }

        
        $data = self::get_nutrition_data($product->get_id());
        if (empty(array_filter($data))) {
            return $tabs; 
        }

        
        $options = get_option('woonutri_options', []);
        $tab_title = isset($options['tab_title']) ? $options['tab_title'] : __('Nutrition Facts', 'smart-nutrition-labels');
        $tab_priority = isset($options['tab_priority']) ? intval($options['tab_priority']) : 50;
        $tab_show_icon = isset($options['tab_show_icon']) ? (bool) $options['tab_show_icon'] : true;

        
        $title = $tab_title;
        if ($tab_show_icon) {
            $title = '<span class="woonutri-tab-icon">🥗</span> ' . $tab_title;
        }

        
        $tabs['woonutri_nutrition'] = [
            'title' => $title,
            'priority' => $tab_priority,
            'callback' => [$this, 'render_nutrition_tab_content'],
        ];

        return $tabs;
    }

    
    public function render_nutrition_tab_content()
    {
        global $product;

        if (!$product) {
            return;
        }

        $product_id = $product->get_id();

        
        echo '<div class="woonutri-tab-content">';

        try {
            echo self::render_label($product_id, [], [
                'position' => 'tab',
                'show_title' => true
            ]);
        } catch (Exception $e) {
            if (current_user_can('manage_options')) {
                echo '<p class="woonutri-error"><!-- WooNutri Error: ' . esc_html($e->getMessage()) . ' --></p>';
            } else {
                echo '<p>' . __('Nutrition information is currently unavailable.', 'smart-nutrition-labels') . '</p>';
            }
        }

        echo '</div>';
    }

    
    public static function shortcode($atts)
    {
        $atts = shortcode_atts([
            'id' => null,
            'theme' => '',
            'position' => 'default',
            'show_title' => true,
            'expandable' => true,
            'show_logo' => null,
            'custom_class' => '',
        ], $atts, 'nutrition_facts');

        $product_id = $atts['id'] ? intval($atts['id']) : get_the_ID();

        if (!$product_id) {
            return '<!-- WooNutri: No product ID found -->';
        }

        try {
            return self::render_label($product_id, [], [
                'theme' => $atts['theme'],
                'position' => $atts['position'],
                'show_title' => filter_var($atts['show_title'], FILTER_VALIDATE_BOOLEAN),
                'expandable' => filter_var($atts['expandable'], FILTER_VALIDATE_BOOLEAN),
                'show_logo' => $atts['show_logo'] !== null ? filter_var($atts['show_logo'], FILTER_VALIDATE_BOOLEAN) : null,
                'custom_class' => sanitize_html_class($atts['custom_class']),
            ]);
        } catch (Exception $e) {
            if (current_user_can('manage_options')) {
                return '<!-- WooNutri Error: ' . esc_html($e->getMessage()) . ' -->';
            }
            return '';
        }
    }

    
    public static function render_label($product_id, $custom = [], $args = [])
    {
        
        if (!empty($custom)) {
            return self::generate_label_html($product_id, $custom, $args);
        }

        
        $product = wc_get_product($product_id);
        if (!$product) {
            return '';
        }

        if (class_exists('Woonutri_Cache')) {
            
            
            
            $timestamp = $product->get_date_modified() ? $product->get_date_modified()->getTimestamp() : 0;
            $args_hash = md5(json_encode($args));
            $cache_key = 'label_' . $product_id . '_' . $args_hash . '_' . $timestamp;

            return Woonutri_Cache::remember($cache_key, function () use ($product_id, $args) {
                return self::generate_label_html($product_id, [], $args);
            }, WEEK_IN_SECONDS);
        }

        
        return self::generate_label_html($product_id, [], $args);
    }

    
    private static function generate_label_html($product_id, $custom = [], $args = [])
    {
        
        $options = self::get_label_options($args);

        
        if (empty($custom)) {
            
            $data = \Woonutri_Meta_Handler::get_nutrition_data_formatted(
                $product_id,
                $options['display_format']
            );
        } else {
            $data = $custom;
        }

        
        if (empty(array_filter($data))) {
            return '';
        }

        

        
        if (!empty($options['nutrient_combination']) && $options['nutrient_combination'] !== 'all') {
            
            if (!class_exists('Woonutri_Nutrient_Combinations')) {
                require_once plugin_dir_path(__FILE__) . '../core/class-nutrient-combinations.php';
            }

            $display_nutrients = \Woonutri_Nutrient_Combinations::get_display_nutrients(
                $options['nutrient_combination'],
                $data
            );

            
            if (!empty($display_nutrients)) {
                $filtered_data = [];
                foreach ($display_nutrients as $nutrient) {
                    $filtered_data[$nutrient['key']] = $nutrient['value'];
                }

                
                
                $preserve_fields = [
                    
                    'allergens',
                    'dietary_preferences',
                    'may_contain',
                    'ingredients',
                    'serving_size',
                    'servings_per_container',
                    'vitamins',
                    'barcode',
                    'nutri_score',
                    'nutri_score_points',

                    
                    'monounsaturated_fat',
                    'polyunsaturated_fat',
                    'trans_fat',
                    'omega_3',
                    'omega_6',
                    'dha',
                    'epa',

                    
                    'cholesterol',
                    'sodium',
                    'potassium',
                    'fiber',
                    'added_sugars',
                    'sugar_alcohols',
                    'starch',

                    
                    'vitamin_a',
                    'vitamin_d',
                    'vitamin_e',
                    'vitamin_k',

                    
                    'vitamin_c',
                    'thiamin',
                    'riboflavin',
                    'niacin',
                    'vitamin_b1',
                    'vitamin_b2',
                    'vitamin_b3',
                    'pantothenic_acid',
                    'vitamin_b6',
                    'biotin',
                    'folate',
                    'vitamin_b12',
                    'choline',

                    
                    'calcium',
                    'iron',
                    'phosphorus',
                    'magnesium',
                    'chloride',

                    
                    'iodine',
                    'zinc',
                    'selenium',
                    'copper',
                    'manganese',
                    'chromium',
                    'molybdenum',
                    'fluoride',

                    
                    'caffeine',
                    'alcohol',
                    'beta_carotene',
                    'lycopene',
                    'lutein'
                ];
                foreach ($preserve_fields as $field) {
                    if (isset($data[$field])) {
                        $filtered_data[$field] = $data[$field];
                    }
                }

                $data = $filtered_data;
            }
        }

        
        $styles = self::get_label_styles($options);

        
        ob_start();

        
        if (!empty($styles['font_link'])) {
            echo $styles['font_link'];
        }

        
        $classes = ['woonutri-label', $options['theme_class'], $options['layout_class']];

        if (!empty($options['position_class'])) {
            $classes[] = $options['position_class'];
        }

        if (!empty($options['custom_class'])) {
            $classes[] = $options['custom_class'];
        }

        if ($options['expandable']) {
            $classes[] = 'woonutri-expandable-enabled';
        }

        $class_attr = implode(' ', array_map('esc_attr', $classes));

        
        ?>
        <div class="<?php echo $class_attr; ?>" dir="auto" style="<?php echo $styles['inline_style']; ?>"
            data-product-id="<?php echo esc_attr($product_id); ?>" role="region"
            aria-label="<?php esc_attr_e('Nutrition Facts', 'smart-nutrition-labels'); ?>">

            <?php
            
            self::render_label_header($options);

            
            self::render_nutri_score($product_id, $options);

            
            self::render_label_content($data, $options);

            
            self::render_allergens_section($data, $options);

            
            self::render_dietary_section($data, $options);

            
            self::render_label_footer($options);
            ?>

        </div>
        <?php

        
        return ob_get_clean();
    }

    
    public static function format_ingredients($ingredients)
    {
        if (empty($ingredients)) {
            return '';
        }

        $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();

        if ($is_pro && class_exists('Woonutri_Allergens')) {
            
            return Woonutri_Allergens::highlight_allergens_in_text($ingredients);
        }

        
        return esc_html($ingredients);
    }

    
    private static function render_allergens_section($data, $options)
    {
        $allergens = isset($data['allergens']) ? (array) $data['allergens'] : [];
        $may_contain = isset($data['may_contain']) ? $data['may_contain'] : '';

        
        if (is_string($allergens) && !empty($allergens)) {
            $allergens = array_map('trim', explode(',', $allergens));
        }

        if (empty($allergens) && empty($may_contain)) {
            return;
        }

        $all_allergens = Woonutri_Allergens::get_eu_allergens();
        ?>
        <div class="woonutri-allergens-section">
            <?php if (!empty($allergens)): ?>
                <div class="woonutri-allergens-contains">
                    <strong><?php _e('Contains:', 'smart-nutrition-labels'); ?></strong>
                    <div class="woonutri-allergen-list">
                        <?php foreach ($allergens as $allergen_key):
                            $allergen_info = isset($all_allergens[$allergen_key]) ? $all_allergens[$allergen_key] : null;
                            if ($allergen_info):
                                $icon = isset($allergen_info['icon']) ? $allergen_info['icon'] : '';
                                $name = $allergen_info['name'];
                            else:
                                $icon = '';
                                $name = ucfirst($allergen_key);
                            endif;
                            ?>
                            <span class="woonutri-allergen-item" data-allergen="<?php echo esc_attr($allergen_key); ?>">
                                <?php if ($icon): ?><span class="allergen-icon"><?php echo esc_html($icon); ?></span><?php endif; ?>
                                <span class="allergen-name"><?php echo esc_html($name); ?></span>
                            </span>
                        <?php endforeach; ?>
                    </div>
                </div>
            <?php endif; ?>

            <?php if (!empty($may_contain)): ?>
                <div class="woonutri-allergens-may-contain">
                    <strong><?php _e('May contain:', 'smart-nutrition-labels'); ?></strong>
                    <span><?php echo esc_html($may_contain); ?></span>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }

    
    private static function render_dietary_section($data, $options)
    {
        $preferences = isset($data['dietary_preferences']) ? (array) $data['dietary_preferences'] : [];

        
        if (is_string($preferences) && !empty($preferences)) {
            $preferences = array_map('trim', explode(',', $preferences));
        }

        if (empty($preferences)) {
            return;
        }

        $all_preferences = Woonutri_Allergens::get_dietary_preferences();
        ?>
        <div class="woonutri-dietary-section-frontend">
            <div class="woonutri-dietary-badges">
                <?php foreach ($preferences as $pref_key):
                    $pref_info = isset($all_preferences[$pref_key]) ? $all_preferences[$pref_key] : null;
                    if ($pref_info):
                        $icon = isset($pref_info['icon']) ? $pref_info['icon'] : '';
                        $name = $pref_info['name'];
                        $type = isset($pref_info['type']) ? $pref_info['type'] : 'positive';
                        ?>
                        <span class="woonutri-dietary-badge badge-<?php echo esc_attr($type); ?>"
                            data-dietary="<?php echo esc_attr($pref_key); ?>"
                            title="<?php echo esc_attr($pref_info['description']); ?>">
                            <?php if ($icon): ?><span class="dietary-icon"><?php echo esc_html($icon); ?></span><?php endif; ?>
                            <span class="dietary-name"><?php echo esc_html($name); ?></span>
                        </span>
                    <?php endif;
                endforeach; ?>
            </div>
        </div>
        <?php
    }

    
    public static function get_label_text($key, $default = '')
    {
        static $options = null;

        if (null === $options) {
            $options = get_option('woonutri_options', []);
        }

        
        $text_map = [
            'title' => 'label_title',
            'per_serving' => 'label_per_serving',
            'per_100g' => 'label_per_100g',
            'daily_value' => 'label_daily_value',
        ];

        $option_key = isset($text_map[$key]) ? $text_map[$key] : '';

        
        if ($option_key && !empty($options[$option_key])) {
            return $options[$option_key];
        }

        
        $defaults = [
            'title' => __('Nutrition Facts', 'smart-nutrition-labels'),
            'per_serving' => __('Per serving', 'smart-nutrition-labels'),
            'per_100g' => __('Per 100g', 'smart-nutrition-labels'),
            'daily_value' => __('% Daily Value*', 'smart-nutrition-labels'),
        ];

        return isset($defaults[$key]) ? $defaults[$key] : $default;
    }

    
    private static function get_nutrition_data($product_id)
    {
        $data = [];

        try {
            $data = \Woonutri_Meta_Handler::get_nutrition_data($product_id);
        } catch (Exception $e) {
            return [];
        }

        return apply_filters('woonutri_nutrition_data', $data, $product_id);
    }

    
    private static function get_label_options($args = [])
    {
        
        $options = get_option('woonutri_options', []);

        
        $defaults = [
            'theme' => isset($options['label_theme']) ? $options['label_theme'] : 'basic',
            'primary_color' => isset($options['primary_color']) ? $options['primary_color'] : '#2a7ae2',
            'font_family' => isset($options['font_family']) ? $options['font_family'] : 'default',
            'custom_font' => isset($options['custom_font']) ? $options['custom_font'] : '',
            'font_size' => isset($options['font_size']) ? $options['font_size'] : 'normal',
            'font_size_custom' => isset($options['font_size_custom']) ? $options['font_size_custom'] : '16',
            'background_color' => isset($options['background_color']) ? $options['background_color'] : '#fff',
            'text_color' => isset($options['text_color']) ? $options['text_color'] : '#222',
            'header_color' => isset($options['header_color']) ? $options['header_color'] : '',
            'border_color' => isset($options['border_color']) ? $options['border_color'] : '#222',
            'border_radius' => isset($options['border_radius']) ? $options['border_radius'] : 'medium',
            'table_layout' => isset($options['table_layout']) ? $options['table_layout'] : 'standard',
            'show_logo' => isset($options['show_logo']) ? (bool) $options['show_logo'] : false,
            'enable_collapsible' => isset($options['enable_collapsible']) ? (bool) $options['enable_collapsible'] : true,
            'position' => 'default',
            'show_title' => true,
            'expandable' => true,
            'custom_class' => '',
            'nutrient_combination' => isset($options['nutrient_combination']) ? $options['nutrient_combination'] : 'basic',
            'show_nutri_score' => isset($options['show_nutri_score']) ? (bool) $options['show_nutri_score'] : true,
            'display_format' => isset($options['display_format']) ? $options['display_format'] : 'per_100g',
        ];

        
        $unit_fields = ['energy_kcal', 'fat', 'sugar', 'protein', 'fiber', 'salt'];
        $defaults['units'] = [];
        foreach ($unit_fields as $field) {
            $defaults['units'][$field] = isset($options['unit_' . $field]) ? $options['unit_' . $field] : ($field === 'energy_kcal' ? 'kcal' : 'g');
        }

        
        foreach ($args as $key => $value) {
            if ($value !== null && $value !== '') {
                $defaults[$key] = $value;
            }
        }

        
        
        $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();
        $is_preview_mode = isset($args['theme']) && wp_doing_ajax() && current_user_can('manage_woocommerce');

        if (!$is_pro && !$is_preview_mode) {
            
            $defaults['theme'] = 'basic';
        }

        
        if (empty($defaults['header_color'])) {
            $defaults['header_color'] = $defaults['primary_color'];
        }

        
        $defaults['theme_class'] = 'woonutri-theme-' . esc_attr($defaults['theme']);
        $defaults['position_class'] = $defaults['position'] !== 'default' ? 'woonutri-position-' . esc_attr($defaults['position']) : '';
        $defaults['layout_class'] = 'woonutri-layout-' . esc_attr($defaults['table_layout']);

        return apply_filters('woonutri_label_options', $defaults, $args);
    }

    
    private static function get_label_styles($options)
    {
        
        $font_css = 'inherit';
        if ($options['custom_font']) {
            $font_css = "'{$options['custom_font']}', sans-serif";
        } else if ($options['font_family'] === 'sans') {
            $font_css = 'Arial, Helvetica, sans-serif';
        } else if ($options['font_family'] === 'serif') {
            $font_css = 'Georgia, Times, serif';
        } else if ($options['font_family'] === 'mono') {
            $font_css = 'Menlo, Monaco, monospace';
        }

        
        $font_size_css = '16px';
        if ($options['font_size'] === 'small') {
            $font_size_css = '14px';
        } else if ($options['font_size'] === 'normal') {
            $font_size_css = '16px';
        } else if ($options['font_size'] === 'large') {
            $font_size_css = '18px';
        } else if ($options['font_size'] === 'custom') {
            $font_size_css = intval($options['font_size_custom']) . 'px';
        }

        
        $radius_css = '8px';
        if ($options['border_radius'] === 'none') {
            $radius_css = '0';
        } else if ($options['border_radius'] === 'small') {
            $radius_css = '4px';
        } else if ($options['border_radius'] === 'medium') {
            $radius_css = '8px';
        } else if ($options['border_radius'] === 'large') {
            $radius_css = '16px';
        }

        
        $padding_modifier = '';
        if ($options['table_layout'] === 'compact') {
            $padding_modifier = '--woonutri-padding-cell: 0.3rem 0.5rem;';
        } else if ($options['table_layout'] === 'spacious') {
            $padding_modifier = '--woonutri-padding-cell: 0.6rem 0.8rem;';
        }

        
        
        $distinctive_themes = [
            
            'dark', 'gradient', 'neon', 'sport', 'keto', 'tech', 'luxury',
            
            'australia', 'canada', 'nutriscore', 'latin', 'uk',
            
            'kids', 'vegan', 'organic', 'retro', 'medical', 'eco',
            'gda', 'artisan', 'paper', 'glass', 'corporate', 'boxed'
        ];

        $theme = isset($options['theme']) ? $options['theme'] : 'basic';
        $is_distinctive = in_array($theme, $distinctive_themes);

        
        if ($is_distinctive) {
            
            $inline_style = sprintf(
                'font-family:%s;font-size:%s;%s',
                esc_attr($font_css),
                esc_attr($font_size_css),
                $padding_modifier
            );
        } else {
            
            $inline_style = sprintf(
                'font-family:%s;font-size:%s;border-radius:%s;--woonutri-primary:%s;--woonutri-bg:%s;--woonutri-text:%s;--woonutri-header:%s;--woonutri-border:%s;%s',
                esc_attr($font_css),
                esc_attr($font_size_css),
                esc_attr($radius_css),
                esc_attr($options['primary_color']),
                esc_attr($options['background_color']),
                esc_attr($options['text_color']),
                esc_attr($options['header_color']),
                esc_attr($options['border_color']),
                $padding_modifier
            );
        }

        
        return [
            'inline_style' => $inline_style,
            'font_link' => '', 
        ];
    }

    
    private static function render_label_header($options)
    {
        $theme = $options['theme'];
        $show_logo = $options['show_logo'];
        $show_title = $options['show_title'];

        
        $logo_themes = ['classic', 'modern', 'fda', 'premium', 'dark'];
        if (in_array($theme, $logo_themes) && $show_logo): ?>
            <div class="woonutri-label-logo">
                <img src="<?php echo esc_url(WOONUTRI_PLUGIN_URL . 'assets/WooNutri_logo.png'); ?>"
                    alt="<?php esc_attr_e('Nutrition Facts Logo', 'smart-nutrition-labels'); ?>" style="max-width: 90px; margin-bottom: 0.5em;" />
            </div>
        <?php endif;

        
        if ($show_title) {
            
            $table_header_themes = ['basic', 'minimal', 'clean', 'simple', 'compact'];

            if (in_array($theme, $table_header_themes)) {
                
            } else {
                
                $header_config = self::get_theme_header_config($theme);
                if ($header_config) {
                    
                    $distinctive_themes = [
                        'dark', 'gradient', 'neon', 'sport', 'keto', 'tech', 'luxury',
                        'australia', 'canada', 'nutriscore', 'latin', 'uk',
                        'kids', 'vegan', 'organic', 'retro', 'medical', 'eco',
                        'gda', 'artisan', 'paper', 'glass', 'corporate', 'boxed'
                    ];
                    $header_style = in_array($theme, $distinctive_themes) ? '' : 'style="color:var(--woonutri-header)"';

                    echo '<div class="woonutri-' . esc_attr($theme) . '-header" ' . $header_style . '>';
                    echo esc_html($header_config['title']);
                    echo '</div>';

                    
                    if (!empty($header_config['subtitle'])) {
                        echo '<div class="woonutri-' . esc_attr($theme) . '-subheader">';
                        echo esc_html($header_config['subtitle']);
                        echo '</div>';
                    }
                }
            }
        }
    }

    
    private static function get_theme_header_config($theme)
    {
        $configs = [
            
            'classic'   => ['title' => __('Nutrition Facts', 'smart-nutrition-labels')],
            'modern'    => ['title' => __('Nutrition Facts', 'smart-nutrition-labels')],
            'fda'       => ['title' => __('Nutrition Facts', 'smart-nutrition-labels')],
            'premium'   => ['title' => __('Nutrition Facts', 'smart-nutrition-labels')],
            'dark'      => ['title' => __('Nutrition Facts', 'smart-nutrition-labels')],
            'eu'        => ['title' => __('Nutrition Information', 'smart-nutrition-labels')],
            'us-fda'    => ['title' => __('Nutrition Facts', 'smart-nutrition-labels'), 'subtitle' => __('Serving Size 1 Container', 'smart-nutrition-labels')],
            'uk'        => ['title' => __('Nutrition Information', 'smart-nutrition-labels')],

            
            'australia'  => ['title' => __('Nutrition Information', 'smart-nutrition-labels')],
            'canada'     => ['title' => __('Nutrition Facts / Valeur nutritive', 'smart-nutrition-labels')],
            'nutriscore' => ['title' => __('Nutrition Information', 'smart-nutrition-labels')],
            'latin'      => ['title' => __('Información Nutricional', 'smart-nutrition-labels')],
            'gda'        => ['title' => __('Guideline Daily Amounts', 'smart-nutrition-labels')],

            
            'corporate' => ['title' => __('Nutrition Facts', 'smart-nutrition-labels')],
            'spacious'  => ['title' => __('Nutrition Facts', 'smart-nutrition-labels')],
            'boxed'     => ['title' => __('Nutrition Facts', 'smart-nutrition-labels')],
            'outline'   => ['title' => __('Nutrition Facts', 'smart-nutrition-labels')],
            'gradient'  => ['title' => __('Nutrition Facts', 'smart-nutrition-labels')],
            'glass'     => ['title' => __('Nutrition Facts', 'smart-nutrition-labels')],
            'paper'     => ['title' => __('Nutrition Facts', 'smart-nutrition-labels')],

            
            'organic'   => ['title' => __('Organic Nutrition', 'smart-nutrition-labels')],
            'sport'     => ['title' => __('Performance Nutrition', 'smart-nutrition-labels')],
            'kids'      => ['title' => __('Nutrition Facts', 'smart-nutrition-labels')],
            'vegan'     => ['title' => __('Plant-Based Nutrition', 'smart-nutrition-labels')],
            'keto'      => ['title' => __('Keto Nutrition', 'smart-nutrition-labels')],
            'medical'   => ['title' => __('Nutrition Information', 'smart-nutrition-labels')],
            'artisan'   => ['title' => __('Nutritional Values', 'smart-nutrition-labels')],
            'eco'       => ['title' => __('Eco Nutrition', 'smart-nutrition-labels')],
            'luxury'    => ['title' => __('Nutrition', 'smart-nutrition-labels')],

            
            'retro'     => ['title' => __('Nutrition Facts', 'smart-nutrition-labels')],
            'tech'      => ['title' => __('NUTRITION DATA', 'smart-nutrition-labels')],
            'neon'      => ['title' => __('Nutrition Facts', 'smart-nutrition-labels')],
        ];

        return isset($configs[$theme]) ? $configs[$theme] : null;
    }

    
    private static function render_label_content($data, $options)
    {
        
        echo '<!-- DEBUG RENDER_LABEL_CONTENT: theme=' . ($options['theme'] ?? 'NOT SET') . ' -->';

        $theme = $options['theme'];
        $standard = self::get_standard_from_theme($theme);

        
        if ($standard) {
            $data = Woonutri_Label_Standards::convert_to_standard($data, $standard);
        }

        if ($theme === 'basic' || $theme === 'minimal' || $theme === 'clean'): ?>
            <?php if ($theme === 'basic'): ?>
                <div class="woonutri-basic-header"><?php _e('Nutrition Facts', 'smart-nutrition-labels'); ?></div>
            <?php endif; ?>
            <table class="woonutri-table" role="table">
                <thead>
                    <tr>
                        <th colspan="2" style="color:var(--woonutri-header)" role="columnheader">
                            <?php
                            if ($theme === 'basic') {
                                
                                $serving_size = isset($data['serving_size']) && $data['serving_size'] ? $data['serving_size'] : '';
                                if ($serving_size) {
                                    echo '<span style="font-size: 13px; font-weight: normal;">' . __('Serving Size', 'smart-nutrition-labels') . ' ' . esc_html($serving_size) . '</span>';
                                } else {
                                    
                                    if (isset($options['display_format'])) {
                                        if ($options['display_format'] === 'per_serving') {
                                            echo '<span style="font-size: 13px; font-weight: normal;">' . __('Per Serving', 'smart-nutrition-labels') . '</span>';
                                        } else {
                                            echo '<span style="font-size: 13px; font-weight: normal;">' . __('Per 100g', 'smart-nutrition-labels') . '</span>';
                                        }
                                    }
                                }
                            } else {
                                $format_text = (isset($options['display_format']) && $options['display_format'] === 'per_serving')
                                    ? __('per serving', 'smart-nutrition-labels')
                                    : __('per 100g', 'smart-nutrition-labels');
                                printf(__('Nutrition Facts (%s)', 'smart-nutrition-labels'), $format_text);
                            }
                            ?>
                        </th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($theme === 'basic'): ?>
                        <tr class="woonutri-dv-header">
                            <td></td>
                            <td style="text-align: right; font-weight: 700; font-size: 12px;">
                                <?php _e('% Daily Value*', 'smart-nutrition-labels'); ?>
                            </td>
                        </tr>
                    <?php endif; ?>
                    <?php self::render_rows($data, $options); ?>
                </tbody>
            </table>
            <?php if ($theme === 'basic'): ?>
                <div class="woonutri-basic-footer">
                    *
                    <?php _e('The % Daily Value (DV) tells you how much a nutrient in a serving of food contributes to a daily diet. 2,000 calories a day is used for general nutrition advice.', 'smart-nutrition-labels'); ?>
                </div>
            <?php endif; ?>
        <?php elseif ($theme === 'eu' || $theme === 'uk'): ?>
            <table class="woonutri-table woonutri-eu-table" role="table">
                <thead>
                    <tr>
                        <th class="woonutri-eu-header-left"><?php _e('Nutrition Information', 'smart-nutrition-labels'); ?></th>
                        <th class="woonutri-eu-header-right">
                            <?php
                            $format_text = (isset($options['display_format']) && $options['display_format'] === 'per_serving')
                                ? __('Per serving', 'smart-nutrition-labels')
                                : __('Per 100g', 'smart-nutrition-labels');
                            echo esc_html($format_text);
                            ?>
                        </th>
                    </tr>
                </thead>
                <tbody>
                    <?php self::render_eu_rows($data, $options, $theme); ?>
                </tbody>
            </table>
        <?php elseif ($theme === 'fda' || $theme === 'us-fda'): ?>
            <table class="woonutri-table" role="table">
                <tbody>
                    <?php self::render_fda_rows($data, $options); ?>
                </tbody>
            </table>
        <?php else: ?>
            <table class="woonutri-table" role="table">
                <tbody>
                    <?php self::render_rows($data, $options); ?>
                </tbody>
            </table>
        <?php endif;
    }

    
    private static function render_label_footer($options)
    {
        $theme = $options['theme'];

        if ($theme === 'fda'): ?>
            <div class="woonutri-fda-footer">
                <?php _e('% Daily Values are based on a 2,000 calorie diet.', 'smart-nutrition-labels'); ?>
            </div>
        <?php endif;

        if ($theme === 'us-fda'): ?>
            <div class="woonutri-us-fda-footer">
                <?php _e('* The % Daily Value (DV) tells you how much a nutrient in a serving of food contributes to a daily diet. 2,000 calories a day is used for general nutrition advice.', 'smart-nutrition-labels'); ?>
            </div>
        <?php endif;

        
        do_action('woonutri_label_footer', $options);
    }

    
    private static function get_standard_from_theme($theme)
    {
        $theme_standards = array(
            'fda' => 'fda',
            'us-fda' => 'fda',
            'eu' => 'eu',
            'uk' => 'uk'
        );

        return isset($theme_standards[$theme]) ? $theme_standards[$theme] : null;
    }

    
    private static function render_fda_rows($data, $options = [])
    {
        $field_mapping = Woonutri_Label_Standards::get_field_mapping('fda');
        $daily_values = Woonutri_Label_Standards::get_daily_values('fda');

        
        echo '<tr class="woonutri-serving-row">';
        echo '<td colspan="2">';
        $serving_size = isset($data['serving_size']) && $data['serving_size'] ? $data['serving_size'] : '1 container (15g)';
        $servings_per = isset($data['servings_per_container']) && $data['servings_per_container'] ? $data['servings_per_container'] : '2';
        echo '<strong>' . __('Serving size', 'smart-nutrition-labels') . '</strong> ' . esc_html($serving_size) . '<br>';
        echo '<strong>' . __('Servings per container', 'smart-nutrition-labels') . '</strong> ' . esc_html($servings_per);
        echo '</td>';
        echo '</tr>';

        
        echo '<tr class="woonutri-amount-header">';
        echo '<td colspan="2"><strong>' . __('Amount per serving', 'smart-nutrition-labels') . '</strong></td>';
        echo '</tr>';

        
        echo '<tr class="woonutri-calories-row">';
        echo '<td>';
        echo '<span class="calories-label">' . __('Calories', 'smart-nutrition-labels') . '</span>';
        echo '</td>';
        echo '<td>';
        $calories = isset($data['energy_kcal']) && $data['energy_kcal'] ? $data['energy_kcal'] : '0';
        echo '<span class="calories-value">' . esc_html($calories) . '</span>';
        echo '</td>';
        echo '</tr>';

        
        echo '<tr class="woonutri-dv-header">';
        echo '<td></td>';
        echo '<td><strong>' . __('% Daily Value*', 'smart-nutrition-labels') . '</strong></td>';
        echo '</tr>';

        
        $fda_nutrients = [
            'fat' => [
                'label' => 'Total Fat',
                'bold' => true,
                'show_dv' => true,
                'children' => [
                    'saturated_fat' => ['label' => 'Saturated Fat', 'show_dv' => true],
                    'trans_fat' => ['label' => 'Trans Fat', 'show_dv' => false]
                ]
            ],
            'cholesterol' => ['label' => 'Cholesterol', 'bold' => true, 'show_dv' => true],
            'sodium' => ['label' => 'Sodium', 'bold' => true, 'show_dv' => true],
            'carbs' => [
                'label' => 'Total Carbohydrate',
                'bold' => true,
                'show_dv' => true,
                'children' => [
                    'fiber' => ['label' => 'Dietary Fiber', 'show_dv' => true],
                    'sugar' => [
                        'label' => 'Total Sugars',
                        'show_dv' => false,
                        'children' => [
                            'added_sugars' => ['label' => 'Includes Added Sugars', 'show_dv' => true]
                        ]
                    ]
                ]
            ],
            'protein' => ['label' => 'Protein', 'bold' => true, 'show_dv' => false, 'thick_border' => true],
            'vitamin_d' => ['label' => 'Vitamin D', 'show_dv' => true],
            'calcium' => ['label' => 'Calcium', 'show_dv' => true],
            'iron' => ['label' => 'Iron', 'show_dv' => true],
            'potassium' => ['label' => 'Potassium', 'show_dv' => true, 'thick_border' => true]
        ];

        
        self::render_fda_nutrient_group($fda_nutrients, $data, 0);

        
        if (isset($data['ingredients']) && $data['ingredients'] !== '') {
            echo '<tr class="woonutri-ingredients-section">';
            echo '<td colspan="2">';
            echo '<div class="woonutri-ingredients-section">';
            echo '<strong>' . __('INGREDIENTS:', 'smart-nutrition-labels') . '</strong> ';
            echo self::format_ingredients($data['ingredients']);
            echo '</div>';
            echo '</td>';
            echo '</tr>';
        }
    }

    
    private static function render_rows($data, $options = [])
    {
        
        echo '<!-- DEBUG RENDER_ROWS START: data keys=' . implode(',', array_keys($data)) . ' -->';

        
        $units = [];
        if (isset($options['units']) && is_array($options['units'])) {
            $units = $options['units'];
        }

        
        $default_units = [
            'energy_kcal' => 'kcal',
            'energy_kj' => 'kJ',
            'fat' => 'g',
            'saturated_fat' => 'g',
            'monounsaturated_fat' => 'g',
            'polyunsaturated_fat' => 'g',
            'trans_fat' => 'g',
            'omega_3' => 'g',
            'omega_6' => 'g',
            'cholesterol' => 'mg',
            'carbs' => 'g',
            'fiber' => 'g',
            'sugar' => 'g',
            'added_sugars' => 'g',
            'protein' => 'g',
            'salt' => 'g',
            'sodium' => 'mg',
            'potassium' => 'mg',
            
            'vitamin_a' => 'mcg',
            'vitamin_c' => 'mg',
            'vitamin_d' => 'mcg',
            'vitamin_e' => 'mg',
            'vitamin_k' => 'mcg',
            'vitamin_b1' => 'mg',
            'vitamin_b2' => 'mg',
            'vitamin_b3' => 'mg',
            'vitamin_b6' => 'mg',
            'vitamin_b12' => 'mcg',
            'folate' => 'mcg',
            'biotin' => 'mcg',
            'pantothenic_acid' => 'mg',
            
            'calcium' => 'mg',
            'iron' => 'mg',
            'phosphorus' => 'mg',
            'magnesium' => 'mg',
            'zinc' => 'mg',
            'selenium' => 'mcg',
            'copper' => 'mg',
            'manganese' => 'mg',
            'chromium' => 'mcg',
            'molybdenum' => 'mcg',
            'iodine' => 'mcg',
            
            'caffeine' => 'mg',
            'alcohol' => 'g',
        ];

        
        $units = array_merge($default_units, $units);

        $theme = isset($options['theme']) ? $options['theme'] : 'basic';
        $standard = self::get_standard_from_theme($theme);

        
        if ($standard && in_array($standard, array('eu', 'uk'))) {
            self::render_standard_rows($data, $options, $standard);
            return;
        }

        
        $basic_fields = [
            'energy_kcal' => ['label' => __('Energy', 'smart-nutrition-labels'), 'type' => 'primary'],
            'fat' => [
                'label' => __('Fat', 'smart-nutrition-labels'),
                'type' => 'primary',
                'children' => [
                    'saturated_fat' => ['label' => __('Saturated Fat', 'smart-nutrition-labels'), 'type' => 'nested']
                ]
            ],
            'carbs' => [
                'label' => __('Carbohydrates', 'smart-nutrition-labels'),
                'type' => 'primary',
                'children' => [
                    'fiber' => ['label' => __('Fiber', 'smart-nutrition-labels'), 'type' => 'nested'],
                    'sugar' => [
                        'label' => __('Sugar', 'smart-nutrition-labels'),
                        'type' => 'nested',
                        'children' => [
                            'added_sugars' => ['label' => __('Added Sugars', 'smart-nutrition-labels'), 'type' => 'nested']
                        ]
                    ]
                ]
            ],
            'protein' => ['label' => __('Protein', 'smart-nutrition-labels'), 'type' => 'primary'],
            'salt' => ['label' => __('Salt', 'smart-nutrition-labels'), 'type' => 'primary'],
        ];

        
        $detailed_fields = [
            
            'saturated_fat' => __('Saturated Fat', 'smart-nutrition-labels'),
            'monounsaturated_fat' => __('Monounsaturated Fat', 'smart-nutrition-labels'),
            'polyunsaturated_fat' => __('Polyunsaturated Fat', 'smart-nutrition-labels'),
            'trans_fat' => __('Trans Fat', 'smart-nutrition-labels'),
            'omega_3' => __('Omega-3', 'smart-nutrition-labels'),
            'omega_6' => __('Omega-6', 'smart-nutrition-labels'),
            'cholesterol' => __('Cholesterol', 'smart-nutrition-labels'),
            'sodium' => __('Sodium', 'smart-nutrition-labels'),
            'potassium' => __('Potassium', 'smart-nutrition-labels'),
            'added_sugars' => __('Added Sugars', 'smart-nutrition-labels'),
        ];

        
        $vitamin_fields = [
            'vitamin_a' => __('Vitamin A', 'smart-nutrition-labels'),
            'vitamin_c' => __('Vitamin C', 'smart-nutrition-labels'),
            'vitamin_d' => __('Vitamin D', 'smart-nutrition-labels'),
            'vitamin_e' => __('Vitamin E', 'smart-nutrition-labels'),
            'vitamin_k' => __('Vitamin K', 'smart-nutrition-labels'),
            'vitamin_b1' => __('Thiamin (B1)', 'smart-nutrition-labels'),
            'vitamin_b2' => __('Riboflavin (B2)', 'smart-nutrition-labels'),
            'vitamin_b3' => __('Niacin (B3)', 'smart-nutrition-labels'),
            'vitamin_b6' => __('Vitamin B6', 'smart-nutrition-labels'),
            'vitamin_b12' => __('Vitamin B12', 'smart-nutrition-labels'),
            'folate' => __('Folate', 'smart-nutrition-labels'),
            'biotin' => __('Biotin', 'smart-nutrition-labels'),
            'pantothenic_acid' => __('Pantothenic Acid', 'smart-nutrition-labels'),
        ];

        
        $mineral_fields = [
            'calcium' => __('Calcium', 'smart-nutrition-labels'),
            'iron' => __('Iron', 'smart-nutrition-labels'),
            'phosphorus' => __('Phosphorus', 'smart-nutrition-labels'),
            'magnesium' => __('Magnesium', 'smart-nutrition-labels'),
            'zinc' => __('Zinc', 'smart-nutrition-labels'),
            'selenium' => __('Selenium', 'smart-nutrition-labels'),
            'copper' => __('Copper', 'smart-nutrition-labels'),
            'manganese' => __('Manganese', 'smart-nutrition-labels'),
            'chromium' => __('Chromium', 'smart-nutrition-labels'),
            'molybdenum' => __('Molybdenum', 'smart-nutrition-labels'),
            'iodine' => __('Iodine', 'smart-nutrition-labels'),
        ];

        
        $other_fields = [
            'caffeine' => __('Caffeine', 'smart-nutrition-labels'),
            'alcohol' => __('Alcohol', 'smart-nutrition-labels'),
        ];

        
        $row_count = 0;
        self::render_basic_nutrient_group($basic_fields, $data, $units, $row_count);

        
        $all_extended_fields = array_merge($detailed_fields, $vitamin_fields, $mineral_fields, $other_fields);

        
        $has_extended = false;
        foreach ($all_extended_fields as $key => $label) {
            if (isset($data[$key]) && $data[$key] !== '' && $data[$key] !== '0') {
                $has_extended = true;
                break;
            }
        }

        
        if ($has_extended && (!isset($options['enable_collapsible']) || $options['enable_collapsible'])) {
            
            echo '<tr class="woonutri-expandable" tabindex="0" role="button" aria-expanded="false" aria-controls="woonutri-details-' . esc_attr($row_count) . '">';
            echo '<td colspan="2">' . esc_html__('Detailed Nutrition Information', 'smart-nutrition-labels') . '</td>';
            echo '</tr>';

            
            echo '<tr class="woonutri-details-row">';
            echo '<td colspan="2" class="woonutri-details" id="woonutri-details-' . esc_attr($row_count) . '" aria-hidden="true">';
            echo '<table class="woonutri-inner-table">';

            
            $has_detailed_section = false;
            foreach ($detailed_fields as $key => $label) {
                if (isset($data[$key]) && $data[$key] !== '' && $data[$key] !== '0') {
                    
                    if (!$has_detailed_section) {
                        echo '<tr class="woonutri-section-header"><td colspan="2"><strong>' . esc_html__('Fats Breakdown', 'smart-nutrition-labels') . '</strong></td></tr>';
                    }
                    $has_detailed_section = true;
                    self::render_nutrition_row($key, $label, $data[$key], $units[$key] ?? 'g', 0, 'detailed');
                }
            }

            
            $has_vitamins = false;
            foreach ($vitamin_fields as $key => $label) {
                if (isset($data[$key]) && $data[$key] !== '' && $data[$key] !== '0') {
                    if (!$has_vitamins) {
                        echo '<tr class="woonutri-section-header"><td colspan="2"><strong>' . esc_html__('Vitamins', 'smart-nutrition-labels') . '</strong></td></tr>';
                        $has_vitamins = true;
                    }
                    self::render_nutrition_row($key, $label, $data[$key], $units[$key] ?? 'mg', 0, 'detailed');
                }
            }

            
            $has_minerals = false;
            foreach ($mineral_fields as $key => $label) {
                if (isset($data[$key]) && $data[$key] !== '' && $data[$key] !== '0') {
                    if (!$has_minerals) {
                        echo '<tr class="woonutri-section-header"><td colspan="2"><strong>' . esc_html__('Minerals', 'smart-nutrition-labels') . '</strong></td></tr>';
                        $has_minerals = true;
                    }
                    self::render_nutrition_row($key, $label, $data[$key], $units[$key] ?? 'mg', 0, 'detailed');
                }
            }

            
            foreach ($other_fields as $key => $label) {
                if (isset($data[$key]) && $data[$key] !== '' && $data[$key] !== '0') {
                    self::render_nutrition_row($key, $label, $data[$key], $units[$key] ?? 'mg', 0, 'detailed');
                }
            }

            
            echo '</table>';
            echo '</td>';
            echo '</tr>';
        } else {
            
            foreach ($all_extended_fields as $key => $label) {
                if (isset($data[$key]) && $data[$key] !== '' && $data[$key] !== '0') {
                    $row_count++;
                    self::render_nutrition_row($key, $label, $data[$key], $units[$key] ?? 'g', $row_count);
                }
            }
        }

        
        if (isset($data['vitamins']) && $data['vitamins'] !== '') {
            $row_count++;
            echo '<tr class="woonutri-vitamins-row" data-nutrient="vitamins">';
            echo '<td role="rowheader">' . esc_html__('Other Vitamins & Minerals', 'smart-nutrition-labels') . '</td>';
            echo '<td role="cell">' . esc_html($data['vitamins']) . '</td>';
            echo '</tr>';
        }

        
        if (isset($data['ingredients']) && $data['ingredients'] !== '') {
            $row_count++;
            echo '<tr class="woonutri-ingredients-row" data-nutrient="ingredients">';
            echo '<td role="rowheader">' . esc_html__('Ingredients', 'smart-nutrition-labels') . '</td>';
            echo '<td role="cell">' . self::format_ingredients($data['ingredients']) . '</td>';
            echo '</tr>';
        }

        
        do_action('woonutri_after_nutrition_rows', $data, $options);
    }

    
    private static function render_standard_rows($data, $options, $standard)
    {
        $field_mapping = Woonutri_Label_Standards::get_field_mapping($standard);
        $show_ri = ($standard === 'eu' || $standard === 'uk');

        
        if ($show_ri) {
            echo '<tr class="woonutri-ri-header">';
            echo '<td></td>';
            echo '<td style="text-align: right; font-size: 0.9em;">' . __('%RI*', 'smart-nutrition-labels') . '</td>';
            echo '</tr>';
        }

        foreach ($field_mapping as $key => $config) {
            if (isset($data[$key]) && $data[$key] !== '') {
                $value = Woonutri_Label_Standards::format_value($data[$key], $key, $standard);
                $label = $config['label'];
                $unit = $config['unit'];
                $indent = isset($config['indent']) ? $config['indent'] : false;
                $same_line = isset($config['same_line']) ? $config['same_line'] : false;
                $show_ri = isset($config['show_ri']) ? $config['show_ri'] : false;

                if ($same_line && $label === '') {
                    
                    continue;
                }

                echo '<tr class="woonutri-' . sanitize_html_class($key) . '-row">';

                
                echo '<td style="';
                if ($indent)
                    echo 'padding-left: 20px; font-style: italic;';
                echo '">';
                echo esc_html($label);
                if ($label)
                    echo ' ';
                echo '<span style="font-weight: normal;">' . esc_html($value) . esc_html($unit);

                
                if ($key === 'energy_kj' && isset($data['energy_kcal'])) {
                    $kcal_value = Woonutri_Label_Standards::format_value($data['energy_kcal'], 'energy_kcal', $standard);
                    echo ' / ' . esc_html($kcal_value) . 'kcal';
                }

                echo '</span></td>';

                
                echo '<td style="text-align: right;">';
                if ($show_ri && $value) {
                    $ri_key = ($key === 'saturated_fat' && $standard === 'uk') ? 'saturates' : $key;
                    $ri = Woonutri_Label_Standards::calculate_daily_value_percentage($data[$key], $ri_key, $standard);
                    if ($ri) {
                        echo esc_html($ri);
                    }
                }
                echo '</td>';

                echo '</tr>';
            }
        }

        
        if ($show_ri) {
            echo '<tr class="woonutri-ri-footnote">';
            echo '<td colspan="2" style="font-size: 0.8em; padding-top: 10px; font-style: italic;">';
            echo __('*Reference intake of an average adult (8400 kJ/2000 kcal)', 'smart-nutrition-labels');
            echo '</td>';
            echo '</tr>';
        }

        
        if (isset($data['ingredients']) && $data['ingredients'] !== '') {
            echo '<tr class="woonutri-ingredients-row" data-nutrient="ingredients">';
            echo '<td colspan="2" style="padding-top: 15px;"><strong>' . esc_html__('Ingredients', 'smart-nutrition-labels') . ':</strong> ' . self::format_ingredients($data['ingredients']) . '</td>';
            echo '</tr>';
        }
    }

    
    private static function render_eu_rows($data, $options, $theme)
    {
        
        $eu_nutrients = [
            'energy' => [
                'label' => $theme === 'uk' ? 'Energy' : 'Energy',
                'show_both' => true, 
                'bold' => true
            ],
            'fat' => [
                'label' => 'Fat',
                'bold' => true,
                'show_ri' => true,
                'children' => [
                    'saturated_fat' => [
                        'label' => $theme === 'uk' ? 'of which saturates' : 'of which saturated',
                        'show_ri' => true
                    ]
                ]
            ],
            'carbs' => [
                'label' => 'Carbohydrate',
                'bold' => true,
                'show_ri' => true,
                'children' => [
                    'sugar' => [
                        'label' => 'of which sugars',
                        'show_ri' => true
                    ]
                ]
            ],
            'fiber' => ['label' => 'Fibre', 'show_ri' => false],
            'protein' => ['label' => 'Protein', 'show_ri' => true],
            'salt' => ['label' => 'Salt', 'show_ri' => true]
        ];

        
        echo '<tr class="woonutri-ri-subheader">';
        echo '<td></td>';
        echo '<td class="woonutri-ri-column"><small>' . __('%RI*', 'smart-nutrition-labels') . '</small></td>';
        echo '</tr>';

        
        self::render_eu_nutrient_group($eu_nutrients, $data, 0, $theme);

        
        echo '<tr class="woonutri-ri-footnote">';
        echo '<td colspan="2">';
        echo '<small>*' . __('Reference intake of an average adult (8400 kJ/2000 kcal)', 'smart-nutrition-labels') . '</small>';
        echo '</td>';
        echo '</tr>';

        
        if (isset($data['ingredients']) && $data['ingredients'] !== '') {
            echo '<tr class="woonutri-ingredients-section">';
            echo '<td colspan="2">';
            echo '<div style="margin-top: 15px;">';
            echo '<strong>' . __('Ingredients:', 'smart-nutrition-labels') . '</strong> ';
            echo self::format_ingredients($data['ingredients']);
            echo '</div>';
            echo '</td>';
            echo '</tr>';
        }

        if (isset($data['allergens']) && $data['allergens'] !== '') {
            echo '<tr class="woonutri-allergens-section">';
            echo '<td colspan="2">';
            echo '<div style="margin-top: 10px;">';
            echo '<strong>' . __('Allergens:', 'smart-nutrition-labels') . '</strong> ';
            echo esc_html($data['allergens']);
            echo '</div>';
            echo '</td>';
            echo '</tr>';
        }
    }

    
    private static function render_eu_nutrient_group($nutrients, $data, $indent_level, $theme)
    {
        foreach ($nutrients as $key => $config) {
            
            if ($key === 'energy') {
                
                $has_kj = isset($data['energy_kj']) && $data['energy_kj'] !== '' && $data['energy_kj'] !== '0';
                $has_kcal = isset($data['energy_kcal']) && $data['energy_kcal'] !== '' && $data['energy_kcal'] !== '0';

                if ($has_kj || $has_kcal) {
                    echo '<tr class="woonutri-energy-row">';
                    echo '<td><strong>' . esc_html($config['label']) . '</strong></td>';
                    echo '<td>';

                    
                    if ($has_kj) {
                        echo esc_html($data['energy_kj']) . 'kJ';
                        if ($has_kcal) {
                            echo ' / ' . esc_html($data['energy_kcal']) . 'kcal';
                        }
                    } elseif ($has_kcal) {
                        
                        $kj_value = round(floatval($data['energy_kcal']) * 4.184);
                        echo esc_html($kj_value) . 'kJ / ' . esc_html($data['energy_kcal']) . 'kcal';
                    }

                    echo '</td>';
                    echo '</tr>';
                }
                continue;
            }

            if (isset($data[$key]) && $data[$key] !== '' && $data[$key] !== '0') {
                $value = $data[$key];
                $label = $config['label'];
                $unit = self::get_eu_unit($key);
                $bold = isset($config['bold']) ? $config['bold'] : false;
                $show_ri = isset($config['show_ri']) ? $config['show_ri'] : false;

                $row_class = 'woonutri-' . sanitize_html_class($key) . '-row';
                if ($indent_level > 0) {
                    $row_class .= ' woonutri-eu-indent-' . $indent_level;
                }

                echo '<tr class="' . esc_attr($row_class) . '">';

                
                echo '<td>';

                
                if ($indent_level > 0) {
                    echo '&nbsp;&nbsp;&nbsp;&nbsp;';
                }

                if ($bold)
                    echo '<strong>';
                echo esc_html($label);
                if ($bold)
                    echo '</strong>';
                echo '</td>';

                
                echo '<td>';
                echo esc_html($value) . esc_html($unit);

                
                if ($show_ri && $value && $value !== '0') {
                    $ri = self::calculate_eu_ri($value, $key, $theme);
                    if ($ri) {
                        echo ' <small>(' . esc_html($ri) . ')</small>';
                    }
                }

                echo '</td>';
                echo '</tr>';

                
                if (isset($config['children']) && is_array($config['children'])) {
                    self::render_eu_nutrient_group($config['children'], $data, $indent_level + 1, $theme);
                }
            }
        }
    }

    
    private static function get_eu_unit($key)
    {
        $units = [
            'energy_kj' => 'kJ',
            'energy_kcal' => 'kcal',
            'sodium' => 'mg'
        ];

        return isset($units[$key]) ? $units[$key] : 'g';
    }

    
    private static function calculate_eu_ri($value, $key, $theme)
    {
        
        $ri_values = [
            'energy_kcal' => 2000,
            'fat' => 70,
            'saturated_fat' => 20,
            'carbs' => 260,
            'sugar' => 90,
            'protein' => 50,
            'salt' => 6
        ];

        if (isset($ri_values[$key]) && is_numeric($value)) {
            $percentage = round((floatval($value) / $ri_values[$key]) * 100);
            return $percentage . '%';
        }

        return '';
    }

    
    private static function render_basic_nutrient_group($nutrients, $data, $units, &$row_count)
    {
        foreach ($nutrients as $key => $config) {
            if (isset($data[$key]) && $data[$key] !== '' && $data[$key] !== '0') {
                $row_count++;
                $label = is_array($config) ? $config['label'] : $config;
                $unit = isset($units[$key]) ? $units[$key] : 'g';

                
                $row_classes = ['woonutri-' . sanitize_html_class($key) . '-row', 'woonutri-main-nutrient'];

                
                if ($key === 'energy_kcal') {
                    $row_classes[] = 'woonutri-calories-row';
                }

                
                if ($key === 'protein') {
                    $row_classes[] = 'woonutri-thick-border-bottom';
                }

                
                echo '<tr class="' . esc_attr(implode(' ', $row_classes)) . '" data-nutrient="' . esc_attr($key) . '" data-row="' . esc_attr($row_count) . '">';
                echo '<td role="rowheader">' . esc_html($label) . '</td>';
                echo '<td role="cell">' . esc_html($data[$key]) . ' ' . esc_html($unit) . '</td>';
                echo '</tr>';

                
                if (is_array($config) && isset($config['children'])) {
                    self::render_basic_children($config['children'], $data, $units, $row_count, 1);
                }
            }
        }
    }

    
    private static function render_basic_children($children, $data, $units, &$row_count, $indent_level)
    {
        foreach ($children as $key => $config) {
            if (isset($data[$key]) && $data[$key] !== '' && $data[$key] !== '0') {
                $row_count++;
                $label = is_array($config) ? $config['label'] : $config;
                $unit = isset($units[$key]) ? $units[$key] : 'g';

                
                $row_classes = [
                    'woonutri-' . sanitize_html_class($key) . '-row',
                    'woonutri-sub-nutrient',
                    'woonutri-indent-' . $indent_level
                ];

                echo '<tr class="' . esc_attr(implode(' ', $row_classes)) . '" data-nutrient="' . esc_attr($key) . '" data-row="' . esc_attr($row_count) . '">';
                echo '<td role="rowheader">' . esc_html($label) . '</td>';
                echo '<td role="cell">' . esc_html($data[$key]) . ' ' . esc_html($unit) . '</td>';
                echo '</tr>';

                
                if (is_array($config) && isset($config['children'])) {
                    self::render_basic_children($config['children'], $data, $units, $row_count, $indent_level + 1);
                }
            }
        }
    }

    
    private static function render_fda_nutrient_group($nutrients, $data, $indent_level = 0)
    {
        foreach ($nutrients as $key => $config) {
            if (isset($data[$key]) && $data[$key] !== '' && $data[$key] !== '0') {
                $value = $data[$key];
                $label = $config['label'];
                $unit = self::get_fda_unit($key);
                $bold = isset($config['bold']) ? $config['bold'] : false;
                $show_dv = isset($config['show_dv']) ? $config['show_dv'] : false;
                $thick_border = isset($config['thick_border']) ? $config['thick_border'] : false;

                $row_class = 'woonutri-' . sanitize_html_class($key) . '-row';
                if ($thick_border) {
                    $row_class .= ' woonutri-thick-border';
                }
                if ($indent_level > 0) {
                    $row_class .= ' woonutri-indent-' . $indent_level;
                }

                echo '<tr class="' . esc_attr($row_class) . '">';

                
                echo '<td>';

                
                for ($i = 0; $i < $indent_level; $i++) {
                    echo '&nbsp;&nbsp;&nbsp;&nbsp;';
                }

                if ($bold)
                    echo '<strong>';
                echo esc_html($label) . ' ' . esc_html($value) . esc_html($unit);
                if ($bold)
                    echo '</strong>';
                echo '</td>';

                
                echo '<td>';
                if ($show_dv && $value && $value !== '0') {
                    $dv = class_exists('Woonutri_Label_Standards') ?
                        Woonutri_Label_Standards::calculate_daily_value_percentage($value, $key, 'fda') :
                        self::calculate_simple_dv($value, $key);
                    if ($dv && $dv !== '0%') {
                        echo '<strong>' . esc_html($dv) . '</strong>';
                    }
                }
                echo '</td>';

                echo '</tr>';

                
                if (isset($config['children']) && is_array($config['children'])) {
                    self::render_fda_nutrient_group($config['children'], $data, $indent_level + 1);
                }
            }
        }
    }

    
    private static function get_fda_unit($key)
    {
        $units = [
            'energy_kcal' => 'kcal',
            'cholesterol' => 'mg',
            'sodium' => 'mg',
            'potassium' => 'mg',
            'vitamin_d' => 'mcg',
            'calcium' => 'mg',
            'iron' => 'mg'
        ];

        return isset($units[$key]) ? $units[$key] : 'g';
    }

    
    private static function calculate_simple_dv($value, $key)
    {
        $daily_values = [
            'fat' => 78,
            'saturated_fat' => 20,
            'cholesterol' => 300,
            'sodium' => 2300,
            'carbs' => 275,
            'fiber' => 28,
            'added_sugars' => 50,
            'protein' => 50,
            'vitamin_d' => 20,
            'calcium' => 1300,
            'iron' => 18,
            'potassium' => 4700
        ];

        if (isset($daily_values[$key]) && is_numeric($value)) {
            $percentage = round((floatval($value) / $daily_values[$key]) * 100);
            return $percentage . '%';
        }

        return '';
    }

    
    private static function render_nutrition_row($key, $label, $value, $unit, $row_num, $type = 'basic', $indent_level = 0)
    {
        $row_class = 'woonutri-' . sanitize_html_class($key) . '-row';

        
        if ($type === 'detailed') {
            $row_class .= ' woonutri-detailed-row';
        } elseif ($type === 'primary') {
            $row_class .= ' woonutri-main-nutrient';
        } elseif ($type === 'nested') {
            $row_class .= ' woonutri-sub-nutrient';
        }

        
        if ($indent_level > 0) {
            $row_class .= ' woonutri-indent-' . $indent_level;
        }

        
        $main_nutrients = ['fat', 'carbs', 'protein'];
        if (in_array($key, $main_nutrients)) {
            $row_class .= ' woonutri-main-nutrient';
        }

        
        if ($key === 'energy_kcal') {
            $row_class .= ' woonutri-calories-row';
        }

        
        if ($key === 'protein') {
            $row_class .= ' woonutri-thick-border-bottom';
        }

        echo '<tr class="' . esc_attr($row_class) . '" data-nutrient="' . esc_attr($key) . '" data-row="' . esc_attr($row_num) . '">';
        echo '<td role="rowheader">' . esc_html($label) . '</td>';
        echo '<td role="cell">' . esc_html($value) . ' ' . esc_html($unit) . '</td>';
        echo '</tr>';
    }

    
    private static function render_nutri_score($product_id, $options)
    {
        
        $product_nutriscore_enabled = get_post_meta($product_id, '_woonutri_nutriscore_enabled', true);

        
        $show_nutri_score = isset($options['show_nutri_score']) ? $options['show_nutri_score'] : true;

        
        if ($product_nutriscore_enabled === 'no') {
            return; 
        } elseif ($product_nutriscore_enabled === 'yes') {
            $show_nutri_score = true; 
        }
        

        if (!$show_nutri_score) {
            return;
        }

        
        $nutriscore_override = get_post_meta($product_id, '_woonutri_nutriscore_override', true);
        $product_type = get_post_meta($product_id, '_woonutri_product_type', true);

        if (!empty($nutriscore_override) && $nutriscore_override !== 'auto') {
            
            $grade_colors = [
                'A' => '#038141',
                'B' => '#85BB2F',
                'C' => '#FECB02',
                'D' => '#EE8100',
                'E' => '#E63946'
            ];

            $grade = $nutriscore_override;
            $color = isset($grade_colors[$grade]) ? $grade_colors[$grade] : '#666';
            $points = null; 
            $details = __('Manually set grade', 'smart-nutrition-labels');
        } else {
            
            $nutri_score_data = \Woonutri_Meta_Handler::get_nutri_score_data($product_id, $product_type ?: 'food');

            if (!$nutri_score_data) {
                return;
            }

            $grade = $nutri_score_data['grade'];
            $color = $nutri_score_data['color'];
            $points = $nutri_score_data['points'];
            $details = $nutri_score_data['details'];
        }

        ?>
        <div class="woonutri-nutri-score"
            style="margin-bottom: 15px; padding: 10px; background: #f8f9fa; border-radius: 8px; text-align: center;">

            <div class="woonutri-nutri-score-title"
                style="font-size: 12px; color: #666; margin-bottom: 8px; text-transform: uppercase; letter-spacing: 1px;">
                <?php _e('Nutri-Score', 'smart-nutrition-labels'); ?>
            </div>

            <div class="woonutri-nutri-score-display"
                style="display: flex; align-items: center; justify-content: center; gap: 8px;">

                <!-- Letter Grades A-E -->
                <div class="woonutri-nutri-score-grades" style="display: flex; gap: 2px;">
                    <?php
                    $grades = ['A', 'B', 'C', 'D', 'E'];
                    $grade_colors = [
                        'A' => '#038141',
                        'B' => '#85BB2F',
                        'C' => '#FECB02',
                        'D' => '#EE8100',
                        'E' => '#E63946'
                    ];

                    foreach ($grades as $g) {
                        $is_current = ($g === $grade);
                        $bg_color = $grade_colors[$g];
                        $opacity = $is_current ? '1' : '0.3';
                        $font_weight = $is_current ? 'bold' : 'normal';
                        $font_size = $is_current ? '16px' : '12px';
                        ?>
                        <div class="woonutri-grade-<?php echo strtolower($g); ?>" style="
                                background-color: <?php echo esc_attr($bg_color); ?>;
                                color: white;
                                width: 24px;
                                height: 24px;
                                display: flex;
                                align-items: center;
                                justify-content: center;
                                font-size: <?php echo $font_size; ?>;
                                font-weight: <?php echo $font_weight; ?>;
                                border-radius: 4px;
                                opacity: <?php echo $opacity; ?>;
                                transition: all 0.3s ease;
                             ">
                            <?php echo $g; ?>
                        </div>
                        <?php
                    }
                    ?>
                </div>

                <!-- Score Details -->
                <div class="woonutri-score-details" style="margin-left: 15px; text-align: left; font-size: 11px; color: #666;">
                    <div><?php printf(__('Score: %d', 'smart-nutrition-labels'), $points); ?></div>
                    <?php if ($details && isset($details['is_beverage'])): ?>
                        <div style="font-style: italic;">
                            <?php echo $details['is_beverage'] ? __('Beverage', 'smart-nutrition-labels') : __('Food', 'smart-nutrition-labels'); ?>
                        </div>
                    <?php endif; ?>
                </div>

            </div>

            <!-- Expandable Details -->
            <?php if ($details): ?>
                <div class="woonutri-nutri-score-expand" style="margin-top: 8px;">
                    <button type="button" class="woonutri-nutri-score-toggle" style="
                                background: none; 
                                border: 1px solid #ddd; 
                                color: #666; 
                                font-size: 10px; 
                                padding: 4px 8px; 
                                border-radius: 4px; 
                                cursor: pointer;
                                text-transform: uppercase;
                                letter-spacing: 0.5px;
                            "
                        onclick="this.nextElementSibling.style.display = this.nextElementSibling.style.display === 'none' ? 'block' : 'none'; this.textContent = this.textContent === '<?php _e('Details', 'smart-nutrition-labels'); ?>' ? '<?php _e('Hide', 'smart-nutrition-labels'); ?>' : '<?php _e('Details', 'smart-nutrition-labels'); ?>';">
                        <?php _e('Details', 'smart-nutrition-labels'); ?>
                    </button>

                    <div class="woonutri-nutri-score-breakdown"
                        style="display: none; margin-top: 8px; font-size: 10px; color: #555; text-align: left;">
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 4px; margin-bottom: 8px;">
                            <div><strong><?php _e('Negative Points:', 'smart-nutrition-labels'); ?></strong>
                                <?php echo intval($details['negative_points'] ?? 0); ?></div>
                            <div><strong><?php _e('Positive Points:', 'smart-nutrition-labels'); ?></strong>
                                <?php echo intval($details['positive_points'] ?? 0); ?></div>
                        </div>

                        <div style="font-size: 9px; line-height: 1.3; color: #777;">
                            <?php
                            $breakdown_details = $details['details'] ?? [];
                            if ($breakdown_details): ?>
                                <div style="margin-bottom: 4px;"><strong><?php _e('Breakdown:', 'smart-nutrition-labels'); ?></strong></div>
                                <div style="display: grid; grid-template-columns: repeat(2, 1fr); gap: 2px;">
                                    <div><?php _e('Energy:', 'smart-nutrition-labels'); ?>
                                        <?php echo intval($breakdown_details['energy_points'] ?? 0); ?>
                                    </div>
                                    <div><?php _e('Sat Fat:', 'smart-nutrition-labels'); ?>
                                        <?php echo intval($breakdown_details['saturated_fat_points'] ?? 0); ?>
                                    </div>
                                    <div><?php _e('Sugar:', 'smart-nutrition-labels'); ?>
                                        <?php echo intval($breakdown_details['sugar_points'] ?? 0); ?>
                                    </div>
                                    <div><?php _e('Sodium:', 'smart-nutrition-labels'); ?>
                                        <?php echo intval($breakdown_details['sodium_points'] ?? 0); ?>
                                    </div>
                                    <div><?php _e('Fiber:', 'smart-nutrition-labels'); ?>
                                        <?php echo intval($breakdown_details['fiber_points'] ?? 0); ?>
                                    </div>
                                    <div><?php _e('Protein:', 'smart-nutrition-labels'); ?>
                                        <?php echo intval($breakdown_details['protein_points'] ?? 0); ?>
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            <?php endif; ?>

        </div>
        <?php
    }
}