<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class Woonutri_Product_Comparison
{

    public static function init()
    {
        add_shortcode('woonutri_compare', [__CLASS__, 'render_comparison_shortcode']);

        add_action('wp_ajax_woonutri_get_comparison', [__CLASS__, 'ajax_get_comparison']);
        add_action('wp_ajax_nopriv_woonutri_get_comparison', [__CLASS__, 'ajax_get_comparison']);

        add_action('woocommerce_after_add_to_cart_button', [__CLASS__, 'render_compare_button'], 20);

        add_action('wp_enqueue_scripts', [__CLASS__, 'enqueue_scripts']);
    }

    public static function enqueue_scripts()
    {
        if (!is_product() && !is_shop() && !is_product_category()) {
            return;
        }

        wp_enqueue_style(
            'woonutri-comparison',
            WOONUTRI_PLUGIN_URL . 'assets/css/comparison.css',
            [],
            WOONUTRI_VERSION
        );

        wp_enqueue_script(
            'woonutri-comparison',
            WOONUTRI_PLUGIN_URL . 'assets/js/comparison.js',
            ['jquery'],
            WOONUTRI_VERSION,
            true
        );

        wp_localize_script('woonutri-comparison', 'woonutri_compare', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('woonutri_compare'),
            'i18n' => [
                'compare' => __('Compare', 'smart-nutrition-labels'),
                'add_to_compare' => __('+ Add to Compare', 'smart-nutrition-labels'),
                'remove' => __('Remove', 'smart-nutrition-labels'),
                'clear_all' => __('Clear All', 'smart-nutrition-labels'),
                'max_products' => __('Maximum 3 products can be compared', 'smart-nutrition-labels'),
            ],
        ]);
    }

    public static function render_comparison_shortcode($atts)
    {
        $atts = shortcode_atts([
            'ids' => '',
            'fields' => 'energy_kcal,fat,saturated_fat,carbohydrates,sugar,fiber,protein,salt',
            'show_image' => 'yes',
            'show_price' => 'yes',
        ], $atts, 'woonutri_compare');

        $product_ids = array_filter(array_map('intval', explode(',', $atts['ids'])));

        if (empty($product_ids)) {
            return '<div class="woonutri-compare-empty">' . esc_html__('No products selected for comparison.', 'smart-nutrition-labels') . '</div>';
        }

        $fields = array_filter(array_map('trim', explode(',', $atts['fields'])));
        $show_image = $atts['show_image'] === 'yes';
        $show_price = $atts['show_price'] === 'yes';

        return self::render_comparison_table($product_ids, $fields, $show_image, $show_price);
    }

    public static function render_comparison_table($product_ids, $fields, $show_image = true, $show_price = true)
    {
        $products = [];
        $nutrition_data = [];

        foreach ($product_ids as $id) {
            $product = wc_get_product($id);
            if (!$product)
                continue;

            $products[$id] = $product;
            $nutrition_data[$id] = self::get_nutrition_data($id);
        }

        if (empty($products)) {
            return '<div class="woonutri-compare-empty">' . esc_html__('No valid products found.', 'smart-nutrition-labels') . '</div>';
        }

        $field_labels = self::get_field_labels();

        ob_start();
        ?>
        <div class="woonutri-comparison-widget">
            <table class="woonutri-comparison-table">
                <thead>
                    <tr>
                        <th class="woonutri-compare-header">
                            <?php esc_html_e('Nutrition Facts', 'smart-nutrition-labels'); ?>
                        </th>
                        <?php foreach ($products as $id => $product): ?>
                            <th class="woonutri-compare-product">
                                <?php if ($show_image): ?>
                                    <div class="woonutri-compare-image">
                                        <?php echo $product->get_image('thumbnail'); ?>
                                    </div>
                                <?php endif; ?>
                                <a href="<?php echo esc_url($product->get_permalink()); ?>" class="woonutri-compare-name">
                                    <?php echo esc_html($product->get_name()); ?>
                                </a>
                                <?php if ($show_price): ?>
                                    <div class="woonutri-compare-price">
                                        <?php echo $product->get_price_html(); ?>
                                    </div>
                                <?php endif; ?>
                            </th>
                        <?php endforeach; ?>
                    </tr>
                </thead>
                <tbody>
                    <?php
                    $row_class = '';
                    foreach ($fields as $field):
                        $row_class = $row_class === '' ? 'even' : '';
                        $label = isset($field_labels[$field]) ? $field_labels[$field] : ucfirst(str_replace('_', ' ', $field));
                        $unit = self::get_field_unit($field);
                        ?>
                        <tr class="<?php echo esc_attr($row_class); ?>">
                            <td class="woonutri-compare-label">
                                <?php echo esc_html($label); ?>
                                <?php if ($unit): ?>
                                    <span class="woonutri-unit">(
                                        <?php echo esc_html($unit); ?>)
                                    </span>
                                <?php endif; ?>
                            </td>
                            <?php
                            $values = [];
                            foreach ($products as $id => $product):
                                $value = isset($nutrition_data[$id][$field]) ? floatval($nutrition_data[$id][$field]) : 0;
                                $values[$id] = $value;
                            endforeach;

                            $is_higher_better = in_array($field, ['protein', 'fiber', 'vitamin_c', 'vitamin_d', 'iron', 'calcium']);
                            $best_id = $is_higher_better ?
                                array_search(max($values), $values) :
                                array_search(min(array_filter($values, function ($v) {
                                    return $v > 0; }) ?: [0]), $values);

                            foreach ($products as $id => $product):
                                $value = $values[$id];
                                $is_best = ($id === $best_id && $value > 0);
                                ?>
                                <td class="woonutri-compare-value <?php echo $is_best ? 'best-value' : ''; ?>">
                                    <span class="value">
                                        <?php echo $value > 0 ? number_format($value, 1) : '-'; ?>
                                    </span>
                                    <?php if ($is_best): ?>
                                        <span class="woonutri-best-badge" title="<?php esc_attr_e('Best value', 'smart-nutrition-labels'); ?>">✓</span>
                                    <?php endif; ?>
                                </td>
                            <?php endforeach; ?>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php
        return ob_get_clean();
    }

    public static function render_compare_button()
    {
        global $product;

        if (!$product)
            return;

        $is_pro = function_exists('woonutri_fs') && woonutri_fs()->can_use_premium_code();
        if (!$is_pro)
            return;

        $product_id = $product->get_id();
        ?>
        <button type="button" class="button woonutri-add-compare" data-product-id="<?php echo esc_attr($product_id); ?>"
            data-product-name="<?php echo esc_attr($product->get_name()); ?>"
            title="<?php esc_attr_e('Add to nutrition comparison', 'smart-nutrition-labels'); ?>">
            <span class="dashicons dashicons-chart-bar"></span>
            <?php esc_html_e('+ Compare', 'smart-nutrition-labels'); ?>
        </button>
        <?php
    }

    public static function ajax_get_comparison()
    {
        check_ajax_referer('woonutri_compare', 'nonce');

        $product_ids = isset($_POST['product_ids']) ?
            array_filter(array_map('intval', $_POST['product_ids'])) : [];

        if (empty($product_ids)) {
            wp_send_json_error(['message' => __('No products selected', 'smart-nutrition-labels')]);
            return;
        }

        if (count($product_ids) > 3) {
            wp_send_json_error(['message' => __('Maximum 3 products can be compared', 'smart-nutrition-labels')]);
            return;
        }

        $fields = [
            'energy_kcal',
            'fat',
            'saturated_fat',
            'carbohydrates',
            'sugar',
            'fiber',
            'protein',
            'salt'
        ];

        $html = self::render_comparison_table($product_ids, $fields, true, true);

        wp_send_json_success(['html' => $html]);
    }

    private static function get_nutrition_data($product_id)
    {
        $profiles = get_post_meta($product_id, '_woonutri_profiles', true);

        if (!empty($profiles) && is_array($profiles) && isset($profiles[0])) {
            return $profiles[0];
        }

        $fields = [
            'energy_kcal',
            'fat',
            'saturated_fat',
            'carbohydrates',
            'sugar',
            'fiber',
            'protein',
            'salt',
            'cholesterol',
            'trans_fat',
            'sodium',
            'vitamin_c',
            'vitamin_d',
            'iron',
            'calcium'
        ];

        $data = [];
        foreach ($fields as $field) {
            $value = get_post_meta($product_id, '_woonutri_' . $field, true);
            if ($value !== '') {
                $data[$field] = $value;
            }
        }

        return $data;
    }

    private static function get_field_labels()
    {
        return [
            'energy_kcal' => __('Energy', 'smart-nutrition-labels'),
            'energy_kj' => __('Energy', 'smart-nutrition-labels'),
            'fat' => __('Fat', 'smart-nutrition-labels'),
            'saturated_fat' => __('Saturated Fat', 'smart-nutrition-labels'),
            'trans_fat' => __('Trans Fat', 'smart-nutrition-labels'),
            'carbohydrates' => __('Carbohydrates', 'smart-nutrition-labels'),
            'sugar' => __('Sugar', 'smart-nutrition-labels'),
            'fiber' => __('Fiber', 'smart-nutrition-labels'),
            'protein' => __('Protein', 'smart-nutrition-labels'),
            'salt' => __('Salt', 'smart-nutrition-labels'),
            'sodium' => __('Sodium', 'smart-nutrition-labels'),
            'cholesterol' => __('Cholesterol', 'smart-nutrition-labels'),
            'vitamin_c' => __('Vitamin C', 'smart-nutrition-labels'),
            'vitamin_d' => __('Vitamin D', 'smart-nutrition-labels'),
            'iron' => __('Iron', 'smart-nutrition-labels'),
            'calcium' => __('Calcium', 'smart-nutrition-labels'),
        ];
    }

    private static function get_field_unit($field)
    {
        $units = [
            'energy_kcal' => 'kcal',
            'energy_kj' => 'kJ',
            'fat' => 'g',
            'saturated_fat' => 'g',
            'trans_fat' => 'g',
            'carbohydrates' => 'g',
            'sugar' => 'g',
            'fiber' => 'g',
            'protein' => 'g',
            'salt' => 'g',
            'sodium' => 'mg',
            'cholesterol' => 'mg',
            'vitamin_c' => 'mg',
            'vitamin_d' => 'μg',
            'iron' => 'mg',
            'calcium' => 'mg',
        ];
        return isset($units[$field]) ? $units[$field] : '';
    }

    public static function get_styles()
    {
        return '
        .woonutri-comparison-widget {
            margin: 20px 0;
            overflow-x: auto;
        }

        .woonutri-comparison-table {
            width: 100%;
            border-collapse: collapse;
            font-size: 14px;
        }

        .woonutri-comparison-table th,
        .woonutri-comparison-table td {
            padding: 12px 15px;
            text-align: center;
            border: 1px solid #e1e1e1;
        }

        .woonutri-comparison-table th {
            background: #f8f9fa;
            font-weight: 600;
        }

        .woonutri-compare-header {
            width: 150px;
            text-align: left !important;
        }

        .woonutri-compare-label {
            text-align: left !important;
            font-weight: 500;
        }

        .woonutri-compare-label .woonutri-unit {
            color: #666;
            font-weight: 400;
            font-size: 12px;
        }

        .woonutri-compare-image img {
            max-width: 80px;
            height: auto;
            border-radius: 4px;
        }

        .woonutri-compare-name {
            display: block;
            margin-top: 8px;
            font-weight: 600;
            color: #333;
            text-decoration: none;
        }

        .woonutri-compare-name:hover {
            color: #0073aa;
        }

        .woonutri-compare-price {
            margin-top: 4px;
            color: #666;
        }

        .woonutri-compare-value {
            font-size: 15px;
        }

        .woonutri-compare-value.best-value {
            background: #d4edda;
            font-weight: 600;
        }

        .woonutri-best-badge {
            display: inline-block;
            margin-left: 4px;
            color: #28a745;
            font-weight: bold;
        }

        .woonutri-comparison-table tr.even td {
            background: #fafafa;
        }

        .woonutri-compare-empty {
            padding: 40px;
            text-align: center;
            background: #f8f9fa;
            border-radius: 8px;
            color: #666;
        }

        .woonutri-add-compare {
            margin-left: 10px;
            display: inline-flex;
            align-items: center;
            gap: 5px;
        }

        .woonutri-add-compare .dashicons {
            font-size: 16px;
            width: 16px;
            height: 16px;
        }
        ';
    }
}
